/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.generators;

import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.Task;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;
import org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO;
import org.ametys.web.repository.page.Page;

/**
 * Task list generator for the simple service.
 */
public class TaskListGenerator extends ServiceableGenerator
{
    /** The project manager */
    protected ProjectManager _projectManager;
    
    /** The project calaog manager */
    protected ProjectsCatalogueManager _projectCatalogManager;
    
    /** The workspace task DAO */
    protected WorkspaceTaskDAO _workspaceTaskDAO;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    private WorkspaceModuleExtensionPoint _moduleEP;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _moduleEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _projectCatalogManager = (ProjectsCatalogueManager) serviceManager.lookup(ProjectsCatalogueManager.ROLE);
        _workspaceTaskDAO = (WorkspaceTaskDAO) serviceManager.lookup(WorkspaceTaskDAO.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        int max = parameters.getParameterAsInteger("max-results", 0);
        
        UserIdentity currentUser = _currentUserProvider.getUser();
        
        Stream<Task> streamTasks = _projectManager.getUserProjects(currentUser)
            .keySet()
            .stream()
            .map(project -> _workspaceTaskDAO.getProjectTasks(project))
            .flatMap(List::stream)
            .filter(t -> !t.isClosed()) // sax only open task
            .filter(t -> t.getAssignments().contains(currentUser)) // sax only task assigned to current user
            .sorted(Comparator.comparing(t -> t.getCreationDate())); // sort by creation date
        
        if (max > 0)
        {
            streamTasks = streamTasks.limit(max);
        }
        
        // SAX the tasks
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "tasks");

        streamTasks.forEach(task ->
        {
            Project project = _projectManager.getParentProject(task);
            this._saxTask(project, task);
        });
        
        XMLUtils.endElement(contentHandler, "tasks");
        contentHandler.endDocument();
    }
    
    /**
     * SAX necessary task properties for the simple task list service.
     * @param project the parent roject
     * @param task The task
     */
    protected void _saxTask(Project project, Task task)
    {
        try
        {
            AttributesImpl attrs = new AttributesImpl();
            
            attrs.addCDATAAttribute("id", task.getId());
            
            String pageId = _getTaskPageId(project);
            if (pageId != null)
            {
                attrs.addCDATAAttribute("pageId", pageId);
            }
            XMLUtils.startElement(contentHandler, "task", attrs);
            
            // Project
            attrs.clear();
            attrs.addCDATAAttribute("id", project.getId());
            String siteUrl = _projectManager.getProjectUrl(project, null);
            if (siteUrl != null)
            {
                attrs.addCDATAAttribute("url", siteUrl);
            }
            XMLUtils.createElement(contentHandler, "project", project.getTitle());
            
            _projectCatalogManager.saxCategory(contentHandler, project, "projectCategory");
            
            task.dataToSAX(contentHandler);
            
            XMLUtils.endElement(contentHandler, "task");
        }
        catch (Exception e)
        {
            throw new RuntimeException("An error occurred while gathering the tasks' information.");
        }
    }
    
    private String _getTaskPageId (Project project)
    {
        WorkspaceModule module = _moduleEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        Set<Page> taskModulePages = _projectManager.getModulePages(project, module);
        if (!taskModulePages.isEmpty())
        {
            return taskModulePages.iterator().next().getId();
        }
        
        return null;
    }
}
