/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.jcr;

import java.time.ZonedDateTime;
import java.util.List;

import javax.jcr.Node;

import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.workspaces.tasks.Task;
import org.ametys.plugins.workspaces.tasks.TasksList;

/**
 * JCR implementation of the tasks list
 */
public class JCRTasksList extends DefaultTraversableAmetysObject<JCRTasksListFactory> implements TasksList
{
    /** Attribute Id */
    public static final String ATTRIBUTE_ID = "id";
    /** Attribute Position */
    public static final String ATTRIBUTE_POSITION = "position";
    /** Attribute Label */
    public static final String ATTRIBUTE_LABEL = "label";
    /** Attribute Color */
    public static final String ATTRIBUTE_COLOR_ID = "colorId";
    /** Attribute Icon */
    public static final String ATTRIBUTE_ICON_ID = "iconId";
    /** Attribute Author */
    public static final String ATTRIBUTE_AUTHOR = "author";
    /** Attribute Creationdate */
    public static final String ATTRIBUTE_CREATIONDATE = "creationDate";
    /** Attribute Lastmodified */
    public static final String ATTRIBUTE_LASTMODIFIED = "lastModified";
    
    /**
     * Default constructor for the JCRTasksList
     * @param node The JCR node
     * @param parentPath The JCR parent path
     * @param factory The factory
     */
    public JCRTasksList(Node node, String parentPath, JCRTasksListFactory factory)
    {
        super(node, parentPath, factory);
    }

    public String getListId()
    {
        return getValue(ATTRIBUTE_ID);
    }
    
    public void setListId(String label)
    {
        setValue(ATTRIBUTE_ID, label);
    }
    
    public Long getPosition()
    {
        return getValue(ATTRIBUTE_POSITION);
    }
    
    public void setPosition(Long position)
    {
        setValue(ATTRIBUTE_POSITION, position);
    }
    
    public String getLabel()
    {
        return getValue(ATTRIBUTE_LABEL);
    }

    public void setLabel(String label)
    {
        setValue(ATTRIBUTE_LABEL, label);
    }

    public String getColor()
    {
        return getValue(ATTRIBUTE_COLOR_ID);
    }
    
    public void setColor(String colorId)
    {
        setValue(ATTRIBUTE_COLOR_ID, colorId);
    }
    
    public String getIcon()
    {
        return getValue(ATTRIBUTE_ICON_ID);
    }
    
    public void setIcon(String iconId)
    {
        setValue(ATTRIBUTE_ICON_ID, iconId);
    }
    
    public List<Task> getTasks()
    {
        return _getFactory().getWorkspaceTasksListDAO().getChildTask(getId());
    }

    public void addTask(Task task)
    {
        task.setTasksListId(getId());
    }
    
    public UserIdentity getAuthor()
    {
        return getValue(ATTRIBUTE_AUTHOR);
    }
    
    public void setAuthor(UserIdentity author)
    {
        setValue(ATTRIBUTE_AUTHOR, author);
    }
    
    public ZonedDateTime getCreationDate()
    {
        return getValue(ATTRIBUTE_CREATIONDATE);
    }
    
    public void setCreationDate(ZonedDateTime creationDate)
    {
        setValue(ATTRIBUTE_CREATIONDATE, creationDate);
    }
    
    public ZonedDateTime getLastModified()
    {
        return getValue(ATTRIBUTE_LASTMODIFIED);
    }
    
    public void setLastModified(ZonedDateTime lastModifiedDate)
    {
        setValue(ATTRIBUTE_LASTMODIFIED, lastModifiedDate);
    }

    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getTaskListModel());
    }
}
