/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.userprefs;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.web.userpref.FOUserPreferencesConstants;

/**
 * Class handling the front-office user preferences for the tasks list service
 */
public class TasksUserPreferencesManager implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = TasksUserPreferencesManager.class.getName();
    
    /** The current user provider */
    private CurrentUserProvider _currentUserProvider;
    
    /** The rendering context handler */
//    private RenderingContextHandler _renderingContextHandler;

    /** The manager of user preferences for the front-office */
    private UserPreferencesManager _userPreferencesManager;

    /** Utility method for JSON strings */
    private JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
//        _renderingContextHandler = (RenderingContextHandler) serviceManager.lookup(RenderingContextHandler.ROLE);
        _userPreferencesManager = (UserPreferencesManager) serviceManager.lookup(UserPreferencesManager.ROLE /* + ".FO" */); //FIXME uncomment
    }
    
    /**
     * Set the preferences of the current user for the filtered events service
     * @param siteName the name of the site
     * @param zoneItemId the id of the zone item
     * @param projects the projects from which gather the tasks
     * @throws UserPreferencesException if an error occurs while setting the user preferences
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public void setUserPreferences(String siteName, String zoneItemId, List<String> projects) throws UserPreferencesException
    {
        // We save user preferences for the front-end users only
        // FIXME to uncomment
//        if (_renderingContextHandler.getRenderingContext().equals(RenderingContext.FRONT))
//        {
        UserIdentity user = _currentUserProvider.getUser();
        if (user != null && StringUtils.isNotEmpty(user.getLogin()) && StringUtils.isNotEmpty(user.getPopulationId()))
        {
            Map<String, String> values = new HashMap<>();
            
            Map<String, String> contextVars = _getContextVars(siteName);
            
            values.put("projects", _jsonUtils.convertObjectToJson(projects));
            
            _userPreferencesManager.setUserPreferences(user, siteName + "/" + zoneItemId, contextVars, values);
        }
//        }
    }
    
    /**
     * Get the preferences of the current user
     * @param siteName the name of the site
     * @param zoneItemId the id of the zone item
     * @return the current user's preferences 
     * @throws UserPreferencesException if an error occurs while retrieving the preferences of the current user
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getUserPreferences(String siteName, String zoneItemId) throws UserPreferencesException
    {
        Map<String, Object> userPreferences = new HashMap<> ();

        UserIdentity user = _currentUserProvider.getUser();
        if (user != null)
        {
            Map<String, String> contextVars = _getContextVars(siteName);
            
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, siteName + "/" + zoneItemId, contextVars);
            
            Object[] projects = new Object[]{null};
            
            if (!unTypedUserPrefs.isEmpty())
            {
                projects = _jsonUtils.convertJsonToArray(_userPreferencesManager.getUserPreferenceAsString(user, siteName + "/" + zoneItemId, contextVars, "projects"));
            }
            
            userPreferences.put("projects", projects);
        }
        
        return userPreferences;
    }
    
    private Map<String, String> _getContextVars(String siteName)
    {
        Map<String, String> contextVars = new HashMap<>();
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName);
        return contextVars;
    }
}
