/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.workflow;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.ZonedDateTime;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.BaseRichTextElementType;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.type.ModelItemType;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * This workflow function initialize the rich-text's content with a configured default value
 *
 */
public class InitContentFunction extends AbstractContentWorkflowComponent implements EnhancedFunction
{
    private static final String __RICHTEXT_ATTRIBUTE_KEY = "attribute";
    private static final String __RICHTEXT_VALUE_KEY = "value";
    private static final String __RICHTEXT_IS_I18N_KEY = "isI18n";
    private static final String __RICHTEXT_I18N_CATALOG_KEY = "i18nCatalogue";
    private static final String __RICHTEXT_ACTIVATE_COMMENTS_KEY = "activateComments";
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _i18nUtils = (I18nUtils) manager.lookup(org.ametys.core.util.I18nUtils.ROLE);
    }
    
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        _logger.info("Performing content initialisation");
        
        try
        {
            WorkflowAwareContent content = getContent(transientVars);
            
            String attributePath = (String) args.get(__RICHTEXT_ATTRIBUTE_KEY);
            if (StringUtils.isEmpty(attributePath))
            {
                attributePath = "content";
            }
            
            boolean isI18N = "true".equals(args.get(__RICHTEXT_IS_I18N_KEY));
            String i18nCatalogue = (String) args.get(__RICHTEXT_I18N_CATALOG_KEY);
            if (StringUtils.isEmpty(i18nCatalogue))
            {
                // Use application as default catalogue
                i18nCatalogue = "application";
            }
            
            String value = (String) args.get(__RICHTEXT_VALUE_KEY);
            if (StringUtils.isBlank(value))
            {
                _logger.warn(String.format("Missing 'value' argument for workflow function %s, the content '%s' will not be initialized with default value", InitContentFunction.class.getName(), content.getId()));
                return;
            }
            
            if (!(content instanceof ModifiableContent))
            {
                _logger.warn(String.format("Content '%s' is not a modifiable content. It will not be initialized with default value", content.getId()));
                return;
            }
            
            String defaultValue = "";
            if (isI18N)
            {
                I18nizableText text = new I18nizableText(i18nCatalogue, value);
                defaultValue = _i18nUtils.translate(text, content.getLanguage());
            }
            else
            {
                defaultValue = value;
            }
            
            if (content.hasDefinition(attributePath))
            {
                ModelItemType type = content.getDefinition(attributePath).getType();
                if (type instanceof BaseRichTextElementType)
                {
                    _setRichText((ModifiableContent) content, attributePath, defaultValue);
                }
                else
                {
                    _setText((ModifiableContent) content, attributePath, defaultValue);
                }
            }
            
            boolean activeComments = "true".equals(args.get(__RICHTEXT_ACTIVATE_COMMENTS_KEY));
            if (activeComments && content.hasDefinition("comment"))
            {
                ((ModifiableContent) content).setValue("comment", true);
            }
            
            // save changes
            content.saveChanges();
            
            if (content instanceof VersionableAmetysObject)
            {
                // create version
                ((VersionableAmetysObject) content).checkpoint();
            }
        }
        catch (IOException e)
        {
            throw new WorkflowException("Unable to set the rich text in the content", e);
        }
    }
    
    private void _setText(ModifiableContent content, String attributePath, String value)
    {
        content.setValue(attributePath, value);
    }
    
    private void _setRichText(ModifiableContent content, String attributePath, String value) throws IOException
    {
        RichText richText = new RichText();
        
        String docbook = _textToDocbook(value);
        ByteArrayInputStream is = new ByteArrayInputStream(docbook.getBytes(StandardCharsets.UTF_8));
        
        richText.setInputStream(is);
        richText.setEncoding("UTF-8");
        richText.setMimeType("text/xml");
        richText.setLastModificationDate(ZonedDateTime.now());
        
        content.setValue(attributePath, richText);
    }
    
    private String _textToDocbook (String value)
    {
        StringBuilder sb = new StringBuilder();
        
        sb.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        sb.append("<article version=\"5.0\" xmlns=\"http://docbook.org/ns/docbook\">");
        
        sb.append("<para>");
        sb.append(value.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll("\r?\n", "<phrase role=\"linebreak\"/>"));
        sb.append("</para>");
        
        sb.append("</article>");
        
        return sb.toString();
    }

    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.PRE;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_INIT_CONTENT_FUNCTION_LABEL");
    }
}
