<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2024 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0"
        xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
        xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
        xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
        xmlns:math="http://exslt.org/math"
        exclude-result-prefixes="math ametys">

    <xsl:import href="service:web://pages/services/common.xsl"/>
    
	<xsl:variable name="header" select="string(ametys:serviceParameter('header')/text())"/>
    <xsl:variable name="resourcesType" select="string(ametys:serviceParameter('type')/text())"/>
    <xsl:variable name="browseFolders" select="ametys:serviceViewParameter('browseFolders')"/>
    
    <!-- Used in common.xsl -->
    <xsl:variable name="common-service-css-class-name">ametys-zone-files</xsl:variable>
    
    
    <xsl:variable name="raw-is-title-visible">        
        <xsl:call-template name="common-service-head-title-content">
            <xsl:with-param name="title" select="$header"/>                    
        </xsl:call-template>
    </xsl:variable>    
    <xsl:variable name="is-title-visible" select="string-length($raw-is-title-visible) &gt; 0"/>
    
    <xsl:variable name="zoneItemId" select="ametys:zoneItemId()"/>
    
    <xsl:template name="common-service-head-css">
        <link type="text/css" rel="stylesheet" href="{ametys:skinURL('zones/index-files/scss/main.css')}" media="screen, print, handheld"/>
    </xsl:template> 
    
    <xsl:template name="common-service-head-js">
        <script type="text/javascript"  src="{$uri-prefix}/plugins/microsoft365/resources/js/AmetysFront/OneDriveConnector.{ametys:lang()}.js"/>
        <script type="text/javascript" src="{ametys:pluginResourceURL('web', 'js/jquery.treeview.min.js')}"><xsl:comment>empty</xsl:comment></script>
        
        <script type="text/javascript">
            const resourcesType_<xsl:value-of select="$uniqueId"/> = "<xsl:value-of select="$resourcesType"/>";
            const browseFolder_<xsl:value-of select="$uniqueId"/> = "<xsl:value-of select="$browseFolders"/>" == "true";
            
            function loadResources_<xsl:value-of select="$uniqueId"/>(el)
            {
                let $folder = $j(el);
                let itemId = $folder.attr("data-item-id") || '';
                let driveId = $folder.attr("data-drive-id") || '';
                let isLoaded = $folder.attr("data-loaded") == 'true';
                
                $folder.find("> button").attr("aria-expanded", $folder.find("> button").attr("aria-expanded") != 'true');
                
                if (!isLoaded)
                {
                    let $ul = $folder.find('> ul');
                    
                    // Display loading spinner
                    $ul.html(`
                    &lt;div class="loading">
                                &lt;span class="fas fa-spinner" aria-hidden="true">&lt;/span>
                                &lt;span class="sr-only"><i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_LOADING" i18n:catalogue="skin.{$skin}"/>&lt;/span>
                    &lt;/div>`);

                    AmetysFront.OneDriveConnector.getFiles('<xsl:value-of select="$zoneItemId"/>', itemId, driveId, function(response) {
                        
                        $folder.attr("data-loaded", true);
                        $ul.html(''); // remove spinner
                        
                        if (response.results)
                        {
                            let hasFolder = false;
                            
                            response.results.forEach((item) => {
                                // file or folder entry
                                $ul.append(`&lt;li data-type="${item.type}" data-item-id="${item.itemId}" data-drive-id="${item.driveId || ""}">
                                        &lt;a target="_blank" href="${item.webUrl}" title="<i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_OPEN" i18n:catalogue="skin.{$skin}"/> «${item.name}» <i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_OPEN_ACCESSIBILITY" i18n:catalogue="skin.{$skin}"/>">
                                            &lt;span aria-hidden="true" class="icon ${item.type == 'file' ? item.faIcon : 'fas fa-folder'}">&lt;/span>
                                            &lt;span class="info">
                                            &lt;span>${item.name}&lt;/span>
                                            ${
                                              resourcesType_<xsl:value-of select="$uniqueId"/> == 'SHARED_FILES' &amp;&amp; item.driveId &amp;&amp; item.driveLabel
                                                ? '&lt;span class="share">&lt;span aria-hidden="true" class="fas fa-user-friends">&lt;/span>' + item.driveLabel + '&lt;/span>'
                                                : ""
                                            }
                                            ${
                                              resourcesType_<xsl:value-of select="$uniqueId"/> == 'RECENT_FILES' ? 
                                                (item.driveId &amp;&amp; item.driveLabel ? 
                                                    '&lt;span class="share">' + (item.driveType == 'user' ? "<i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_SHARED_OWNER" i18n:catalogue="skin.{$skin}"/>" + item.driveLabel : item.driveLabel) + '&lt;/span>'
                                                    : '&lt;span class="share"><i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_MY_FILES" i18n:catalogue="skin.{$skin}"/>&lt;/span>')
                                                : ''
                                            }
                                            &lt;/span> 
                                        &lt;/a>
                                        ${
                                          browseFolder_<xsl:value-of select="$uniqueId"/> &amp;&amp; item.hasChildren
                                            ? '&lt;ul>&lt;/ul>'
                                            : ""
                                        }
                                    &lt;/li>`);
                                    
                                 hasFolder = hasFolder || item.type == "folder";
                            });
                            
                            if (browseFolder_<xsl:value-of select="$uniqueId"/> &amp;&amp; hasFolder)
                            {
                                $ul.treeview({ collapsed: true });
                                $ul.find('> li.expandable button').attr('aria-expanded', false);
                                $ul.find('> li.expandable button, > li.expandable span').on('click', function(e) {
                                    loadResources_<xsl:value-of select="$uniqueId"/>($j(this).closest('[data-type=folder]'));
                                });
                            }
                        }
                        else if (response.error)
                        {
                            $ul.html(`
                            &lt;div class="error">
                                        ${response.error == 'user-unknown' ? "<i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_ERROR_UNKNOWN_USER" i18n:catalogue="skin.{$skin}"/>" : "<i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_ERROR_REQUEST" i18n:catalogue="skin.{$skin}"/>"}
                            &lt;/div>`);
                        }
                    });
                }
            }
            
            function searchFiles_<xsl:value-of select="$uniqueId"/>(el)
            {
                let $wrapper = $j("#attachments_<xsl:value-of select="$uniqueId"/> .form-group");
                let $searchInput = $wrapper.find("input");
                let $ul = $wrapper.find(".search-results ul");
                let pattern = $searchInput.val().toLowerCase();
                
                $wrapper.find(".search-results").show();
                $wrapper.find(".search-results .loading").show();
                $wrapper.find(".search-results .no-result").hide();
                $searchInput.next().show();
                
                // Close dropdown when ESCAPE key is pressed
                $wrapper.find(".search-results").get(0).addEventListener('keydown', hideDropdownResultOnEscape_<xsl:value-of select="$uniqueId"/>);
                
                AmetysFront.OneDriveConnector.searchFiles('<xsl:value-of select="ametys:zoneItemId()"/>', pattern, function(response) {
                    $ul.html('');
                    $wrapper.find(".search-results .loading").hide();
                    
                    if (response.results &amp;&amp; response.results.length > 0)
                    {
                        response.results.forEach((item) => {
                            $ul.append(`&lt;li>
                                        &lt;a target="_blank" href="${item.webUrl}" title="<i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_DOWNLOAD" i18n:catalogue="skin.{$skin}"/> «${item.name}» <i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_OPEN_ACCESSIBILITY" i18n:catalogue="skin.{$skin}"/>">
                                            &lt;span aria-hidden="true" class="icon ${item.faIcon}">&lt;/span>
                                            &lt;span class="info">
                                            &lt;span>${item.name}&lt;/span>
                                            &lt;/span> 
                                        &lt;/a>
                                    &lt;/li>`);
                        });
                        // Focus first result
                        $wrapper.find(".search-results ul li a").first().trigger('focus');
                    }
                    else // No results or error
                    {
                        $wrapper.find(".search-results .no-result").show();
                    }
                    
                    if (response["has-more"])
                    {
                        let $more = $wrapper.find(".search-results .more-results");
                        $more.find("a").attr("href", response["has-more"]);
                        $more.show();
                    }
                    else
                    {
                        $wrapper.find(".search-results .more-results").hide();
                    }
                });
            }
            
            const hideDropdownResultOnEscape_<xsl:value-of select="$uniqueId"/> = (event) => {
                if (event.key === "Escape") {
                  closeDropdownResults_<xsl:value-of select="$uniqueId"/>();
                }
            };
            
            function closeDropdownResults_<xsl:value-of select="$uniqueId"/>(el)
            {
                let $wrapper = $j("#attachments_<xsl:value-of select="$uniqueId"/> .form-group");
                $wrapper.find(".search-results").hide();
                let $searchInput = $wrapper.find("input").trigger('focus');
            }
                
            function clearSearchFiles_<xsl:value-of select="$uniqueId"/>(el)
            {
                let $wrapper = $j("#attachments_<xsl:value-of select="$uniqueId"/> .form-group");
                let $searchInput = $wrapper.find("input");
                $searchInput.val("");
                $searchInput.next("button").hide();
                $wrapper.find(".search-results").hide();
            }
                
            $j(function () {
                // Build tree view
                $j("#attachments_<xsl:value-of select="$uniqueId"/>").treeview({ collapsed: true });
                
                // Load root resources
                loadResources_<xsl:value-of select="$uniqueId"/>($j('#attachments_<xsl:value-of select="$uniqueId"/>'));
                
                // Search input
                let keyUpTimer_<xsl:value-of select="$uniqueId"/> = null;
                $j("#attachments_<xsl:value-of select="$uniqueId"/> .form-group input").on('keyup', function(event) {
                    if (event.key === "Escape") {
                        return;
                    }
                    // clear the previous timer
                    clearTimeout(keyUpTimer_<xsl:value-of select="$uniqueId"/>);
                    
                    keyUpTimer_<xsl:value-of select="$uniqueId"/> = setTimeout(() => {
                        let pattern = $(this).val();
                        if ($j(this).val().length >= 3) {
                            searchFiles_<xsl:value-of select="$uniqueId"/>();
                        } 
                        else if (pattern == '')
                        {
                            clearSearchFiles_<xsl:value-of select="$uniqueId"/>();
                        }
                    }, 300);
                });
                
                // Close results dropdown on click outside
                $j(window).on('click', function(e) {
                    const el = e.target;
                    if ($j(el).parents('.search-results').length == 0){ 
                        let $wrapper = $j("#attachments_<xsl:value-of select="$uniqueId"/> .form-group");
                        $wrapper.find(".search-results").hide();
                    }
                } );
                $j(window).on('keyup', function(e) {
                    const el = e.target;
                    if( e.which == 9 /* TAB */) {
                        if ($j(el).parents('.search-results').length == 0){ 
                            let $wrapper = $j("#attachments_<xsl:value-of select="$uniqueId"/> .form-group");
                            $wrapper.find(".search-results").hide();
                        }
                    }
                });
            });
        </script>
    </xsl:template> 
      
    <xsl:template name="common-service-head-title">
        <xsl:call-template name="common-service-head-title-content">
            <xsl:with-param name="title" select="$header"/>
        </xsl:call-template>
    </xsl:template>
    
    <xsl:template name="common-service-body-nonempty">
        <div class="ametys-cms-service {$common-service-css-class-name} wrapper">
            <div class="card mb-4">
                <xsl:call-template name="common-service-body-nonempty-content-title"/>
            
                <xsl:call-template name="common-service-body-nonempty-content-content"/>
            </div>
        </div>
    </xsl:template>   
    
    <xsl:template name="common-service-body-nonempty-content-title">
        <xsl:if test="$header">
            <header class="card-header card-header--space-between flex-wrap">
                <h1 class="card-title">
                    <xsl:call-template name="card-header-icon"/>
                    <xsl:value-of select="$header"/>
                </h1>
            </header>
        </xsl:if>
   </xsl:template>
   
   <xsl:template name="card-header-icon">
        <span aria-hidden="true" class="card-header__icon fas fa-cloud-download-alt"></span>
   </xsl:template>
    
    <xsl:template name="common-service-body-nonempty-content-content">
        <div class="card-body pt-0 pb-4 px-0">
            <div class="attachments" id="attachments_{$uniqueId}">
                <xsl:if test="$resourcesType = 'USER_FILES'">
                    <xsl:call-template name="search-form"/>
                </xsl:if>
                <ul data-resources-type="{$resourcesType}">
                    <!-- Resources tree will be insert here -->
                </ul>
            </div>
        </div>
    </xsl:template>   
    
    <xsl:template name="search-form">
        <div class="form-group">
        
            <span aria-hidden="true" class="material-icons search"></span>
            
            <input class="form-control" type="text" placeholder="skin.{$skin}:SKIN_SERVICE_ONEDRIVE_SEARCH" title="skin.{$skin}:SKIN_SERVICE_ONEDRIVE_SEARCH" i18n:attr="title placeholder"/>
            
            <button type="button" aria-label="skin.{$skin}:SKIN_SERVICE_ONEDRIVE_SEARCH_CLEAR" onclick="clearSearchFiles_{$uniqueId}()" style="display: none">
                <span class="fas fa-times" aria-hidden="true"></span>
            </button>
    
            <div class="search-results" style="display: none">
               <div class="loading" style="display: none">
                    <span class="fas fa-spinner" aria-hidden="true"></span>
                    <span class="sr-only"><i18n:text i18n:key="SKIN_SERVICE_ONEDRIVE_LOADING" i18n:catalogue="skin.{$skin}"/></span>
                </div>
                
                <xsl:call-template name="search-no-result"/>
                
                <ul>
                    <!-- Results will be inserted here -->
                </ul>
                
                <div class="more-results" style="display: none">
                    <a href="">Afficher plus de résultats</a>
                </div>
            </div>
        </div>
    
    </xsl:template>
    
    <xsl:template name="search-no-result">
        <p class="no-result" style="display: none">Aucun résultats ne correspond à votre recherche</p>
    </xsl:template>
    
</xsl:stylesheet>       
