<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2023 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0"
        xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
        xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
        xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
        xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
        xmlns:math="java.lang.Math"
        xmlns:url="org.ametys.core.util.URIUtils"
        xmlns:filename="org.ametys.core.util.FilenameUtils"
        xmlns:escaper="org.apache.commons.lang.StringEscapeUtils"
        xmlns:stringutils="org.apache.commons.lang.StringUtils"
        exclude-result-prefixes="ametys resolver escaper stringutils url filename"> 
    
    
    <!-- The number of facet items above which a search filter is available -->
    <xsl:variable name="configuredDropdownSearchFilterLimit" select="ametys:serviceViewParameter('dropdownSearchFilterLimit')"/>
    <xsl:variable name="dropdownSearchFilterLimit">
        <xsl:choose>
            <xsl:when test="$configuredDropdownSearchFilterLimit != ''"><xsl:value-of select="$configuredDropdownSearchFilterLimit"/></xsl:when>
            <xsl:otherwise>10</xsl:otherwise><!-- default value -->
        </xsl:choose>
    </xsl:variable>
    
    <!-- The number of facets dropdown visible by default (when there are 2 or more facets)-->
    <xsl:variable name="configuredMaxVisibleFilters" select="ametys:serviceViewParameter('maxVisibleFilters')"/>
    <xsl:variable name="maxVisibleFilters">
        <xsl:choose>
            <xsl:when test="$configuredMaxVisibleFilters != ''"><xsl:value-of select="$configuredMaxVisibleFilters"/></xsl:when>
            <xsl:otherwise>3</xsl:otherwise><!-- default value -->
        </xsl:choose>
    </xsl:variable>
    
    <!-- The number of facet elements visible by default (when there is 1 facet only)-->
    <xsl:variable name="configuredMaxVisibleFacetItems" select="ametys:serviceViewParameter('maxVisibleFacetItems')"/>
    <xsl:variable name="maxVisibleFacetItems">
        <xsl:choose>
            <xsl:when test="$configuredMaxVisibleFacetItems != ''"><xsl:value-of select="$configuredMaxVisibleFacetItems"/></xsl:when>
            <xsl:otherwise>5</xsl:otherwise><!-- default value -->
        </xsl:choose>
    </xsl:variable>
    
    <xsl:template name="facet-filter">
        <xsl:variable name="nbFacets" select="count(/search/form/facets/facet)"/>
        
        <xsl:if test="$nbFacets > 0">
            <xsl:choose>
                <xsl:when test="$nbFacets = 1">
                    <xsl:call-template name="facet-filter-single"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:call-template name="facet-filter-multiple"/>
                </xsl:otherwise>
            </xsl:choose>
            <xsl:call-template name="facet-filter-js"/>
        </xsl:if>
     </xsl:template>
     
     <!-- Template when search service is configured with only 1 facet -->
     <xsl:template name="facet-filter-single">
        <xsl:if test="/search/form/facets/facet[@total > 0]">
            <div class="filters-all">
                <div class="filters-links" data-ametys-search-role="facet-filter" data-ametys-search-facet-name="{/search/form/facets/facet/@name}">
                    <xsl:for-each select="/search/form/facets/facet[@total > 0]/item[@count > 0]">
                        <xsl:sort select="@count" data-type="number" order="descending"/>
                        <div class="field">
                            <xsl:call-template name="facet-filter-checkbox"/>
                            <xsl:call-template name="facet-filter-label"/>
                        </div>
                    </xsl:for-each>
                    
                    <xsl:call-template name="facet-filter-show-more"/>
                    <xsl:call-template name="facet-filter-show-less"/>
                </div>
            </div>
            
            <div class="filters-all-dropdown only-mobile">
                <div class="filters-links">
                    <xsl:for-each select="/search/form/facets/facet[@total > 0]">
                        <div class="field">
                            <xsl:call-template name="facet-filter-name"/>
                            <xsl:call-template name="facet-filter-list"/>
                        </div>
                    </xsl:for-each>
                </div>
            </div>
        </xsl:if>
     </xsl:template>
     
     <!-- Template when search service is configured with 2 or more facets -->
     <xsl:template name="facet-filter-multiple">
        <xsl:if test="/search/form/facets/facet[@total > 0]">
            <xsl:call-template name="facet-filter-title"/>
            
            <div class="filters-all-dropdown">
                <div class="filters-links">
                    <xsl:for-each select="/search/form/facets/facet[@total > 0]">
                        <div class="field">
                            <xsl:call-template name="facet-filter-name"/>
                            <xsl:call-template name="facet-filter-list"/>
                        </div>
                    </xsl:for-each>
                    
                    <xsl:call-template name="facet-filter-show-more"/>
                    <xsl:call-template name="facet-filter-show-less"/>
                </div>
            </div>
            
            <!-- TODO  results-header-filter-mobile -->
        </xsl:if>
     </xsl:template>     
     <xsl:template name="facet-filter-show-more">
        <a
          href="javascript:void(0)"
          onclick="moreFilters()"
          class="more-filters">
          <i class="fas fa-sliders-h"></i> <i18n:text i18n:key="SKIN_SERVICE_SEARCH_SHOW_MORE_FILTERS" i18n:catalogue="skin.{$skin}"/>
        </a>
     </xsl:template>
     
      <xsl:template name="facet-filter-show-less">
        <a
          href="javascript:void(0)"
          onclick="lessFilters()"
          class="less-filters">
          <i class="fas fa-sliders-h"></i> <i18n:text i18n:key="SKIN_SERVICE_SEARCH_SHOW_LESS_FILTERS" i18n:catalogue="skin.{$skin}"/></a>
     </xsl:template>
     
     <xsl:template name="facet-filter-title">
        <span class="filters-title"><i18n:text i18n:key="SKIN_SERVICE_SEARCH_FILTERS" i18n:catalogue="skin.{$skin}"/></span>
     </xsl:template>
     
     <xsl:template name="facet-filter-name">
        <div class="label">
            <xsl:variable name="count" select="count(item[@selected='true'])"/>
            <label>
                <xsl:if test="$count > 0"><xsl:attribute name="class">active</xsl:attribute></xsl:if>
                <xsl:value-of select="label"/>
                <xsl:if test="$count > 0"><span class="count"><xsl:value-of select="$count"/></span></xsl:if>
            </label>
        </div>
     </xsl:template>
     
     <xsl:template name="facet-filter-label">
        <div class="label">
            <label>
                <xsl:if test="@selected = 'true'"><xsl:attribute name="class">active</xsl:attribute></xsl:if>
                
                <xsl:choose>
                    <xsl:when test="text() != ''">
                        <xsl:value-of select="text()"/> 
                    </xsl:when>
                    <xsl:when test="contains(parent::facet/@name, '$systemProperty$tags')">
                        <!-- CMS-10799 Tag facets has no label (for version 4.6.0 or lower)-->
                        <xsl:value-of select="ametys:tagLabel(ametys:site(), @value, ametys:lang())"/>  
                    </xsl:when>
                </xsl:choose>
                <xsl:text> (</xsl:text><xsl:value-of select="@count"/>)
            </label>
        </div>
     </xsl:template>
      
     <xsl:template name="facet-filter-checkbox">
        <xsl:variable name="id" select="concat(@name, '-', substring-after(math:random(), '.'))"/>
        
        <xsl:call-template name="facet-filter-element-checkbox">
           <xsl:with-param name="selected" select="@selected"/>
           <xsl:with-param name="id" select="$id"/>
       </xsl:call-template>
     </xsl:template>
     
     <xsl:template name="facet-filter-list">
        <div class="field-dropdown" data-ametys-search-role="facet-filter" data-ametys-search-facet-name="{current()/@name}">
            <xsl:call-template name="facet-filter-search"/>
            
            <div class="field-dropdown-list">
                <xsl:call-template name="facet-filter-element-all"/>
                <xsl:call-template name="facet-filter-elements"/>
            </div>
        </div>
     </xsl:template>
     
     <xsl:template name="facet-filter-element-all">
        <div class="field-dropdown-item">
            <xsl:variable name="onclick"><xsl:call-template name="search-js-facet"/></xsl:variable>
            <input type="checkbox" name="{@name}" value="" id="{@name}__all" data-unselected-all-facets="{@name}" data-onclick="{$onclick}">
                <xsl:if test="count(item[@selected='true']) = 0"><xsl:attribute name="checked">checked</xsl:attribute></xsl:if>
            </input>
            <label for="{@name}__all"><i18n:text i18n:key="SKIN_SERVICE_DROPDOWN_ALL_OPTION" i18n:catalogue="skin.{$skin}"/></label>
        </div>
     </xsl:template>
     
     <xsl:template name="facet-filter-search">
        <xsl:if test="count(item[@count > 0]) > $dropdownSearchFilterLimit">
            <div class="field-dropdown-search">
              <input type="search" placeholder="skin.{$skin}:SKIN_SERVICE_SEARCH_DROPDOWN_FILTER_PLACEHOLDER" i18n:attr="placeholder"/>
            </div>
        </xsl:if>
     </xsl:template>
     
     <xsl:template name="facet-filter-element">
        <xsl:variable name="id" select="concat(../@name, '-', substring-after(math:random(), '.'))"/>
        
        <div class="field-dropdown-item">
           <xsl:call-template name="facet-filter-element-checkbox">
               <xsl:with-param name="selected" select="@selected"/>
               <xsl:with-param name="id" select="$id"/>
           </xsl:call-template>
           <label for="{$id}">
                <xsl:choose>
                    <xsl:when test="text() != ''">
                        <xsl:value-of select="text()"/> 
                    </xsl:when>                                    
                    <xsl:when test="contains(parent::facet/@name, '$systemProperty$tags')">
                        <!-- CMS-10799 Tag facets has no label (for version 4.6.0 or lower)-->
                        <xsl:value-of select="ametys:tagLabel(ametys:site(), @value, ametys:lang())"/>  
                    </xsl:when>
                </xsl:choose>
                <xsl:text> </xsl:text>
                <span class="count-dropdown">(<xsl:value-of select="@count"/>)</span>
            </label>
       </div>
    </xsl:template>
    
    <xsl:template name="facet-filter-js">
        <script>
            function moreFilters() {
                $j(".field.hidden")
                  .removeClass("hidden")
                  .addClass("visible");
                $j(".more-filters").hide();
                $j(".less-filters").show();
            }

            function lessFilters() {
                $j(".field.visible")
                  .removeClass("visible")
                  .addClass("hidden");
                $j(".more-filters").show();
                $j(".less-filters").hide();
            }
                      
            $j(document).ready(function () {
                var nbDropdownFacetsVisible = $j(".filters-all-dropdown .field:visible").length;
                var maxVisibleFilters = <xsl:value-of select="$maxVisibleFilters"/>;
                
                if (nbDropdownFacetsVisible > maxVisibleFilters) {
                    // Hide other facets
                    $j(".filters-all-dropdown .field:gt(" + (maxVisibleFilters - 1) + ")").addClass("hidden");
                    // Show more filters button
                    $j(".more-filters").show();
                }
                
                var nbFacetElementsVisible = $j(".filters-all .filters-links .field:visible").length;
                var maxVisibleFacetItems = <xsl:value-of select="$maxVisibleFacetItems"/>;
                if (nbFacetElementsVisible > maxVisibleFacetItems) {
                    // Hide other facets items
                    $j(".filters-all .filters-links .field:gt(" + (maxVisibleFacetItems - 1) + ")").addClass('hidden');
                    // Show more filters button
                    $j(".more-filters").show();
                }
                
                <xsl:if test="count(/search/form/facets/facet) = 1">
                // Trigger click event on facet input when facet's label is clicked (desktop mode)
                $j(".filters-links .field .label").click(function (e) {
                
                    var $checkbox = $j(this).prev("input"),
                        $currentFacetFilter = $checkbox.closest('[data-ametys-search-role=facet-filter]'),
                        facetValue = $checkbox.attr('value'),
                        wasNotSelected = $checkbox.attr('data-ametys-search-role') === 'facet-unselected';
                
                    // Reflect desktop facet item's value on same facet item for mobile mode
                    var $mobileFacetElmt = $j('.filters-all-dropdown.only-mobile').find('input[type="checkbox"][value="' + facetValue + '"]');
                    $mobileFacetElmt.attr('data-ametys-search-role', wasNotSelected ? 'facet-selected' : 'facet-unselected');
                    
                    // Trigger click event on facet input (launch search)
                    $j(this).prev("input").trigger("click");
                });
                
                // This event will be captured before the click event of the facet element (mobile mode)
                document.addEventListener("click", function(e) {
                   var $targetElement = $j(e.target || e.srcElement),
                       $parent = $targetElement.parent();
                   if ($targetElement.is('input[type="checkbox"][data-ametys-search-role]') &amp;&amp; $parent.hasClass("field-dropdown-item"))
                   {
                        // Reflect mobile facet item's value on same facet item for desktop mode
                        var wasNotSelected = $targetElement.attr('data-ametys-search-role') === 'facet-unselected',
                            facetValue = $targetElement.attr('value');
                            
                        var $desktopFacetElmt = $j('.filters-all .filters-links').find('input[type="checkbox"][value="' + facetValue + '"]');
                        $desktopFacetElmt.attr('data-ametys-search-role', wasNotSelected ? 'facet-selected' : 'facet-unselected');
                   }
                }, true);
                </xsl:if>

                $j(".filters-all-dropdown .label").click(function (e) {
                  const labelPosition = $j(this).offset();
                  const labelWidth = $j(this).width();

                  $j(this).parent().find(".field-dropdown").toggle();
                  // check if field dropdown is outside screen
                  const fieldDropdown = $j(this)
                    .parent()
                    .find(".field-dropdown");
                  const fieldDropdownOffset = fieldDropdown.offset();
                  const fieldDropdownWidth = fieldDropdown.width();
                  const fieldDropdownHeight = fieldDropdown.height();
                  const windowWidth = $j(window).width();
                  const windowHeight = $j(window).height();
                  const windowScrollTop = $j(window).scrollTop();
                  const windowScrollLeft = $j(window).scrollLeft();

                  if (
                    fieldDropdownOffset.left + fieldDropdownWidth >
                    windowWidth + windowScrollLeft
                  ) {
                    fieldDropdown.css({
                      left: "initial",
                      right:
                        windowWidth -
                        labelPosition.left -
                        labelWidth -
                        50 +
                        "px",
                    });
                  }

                  // hide other field dropdown
                  $j(".filters-all-dropdown .field-dropdown")
                    .not($j(this).parent().find(".field-dropdown"))
                    .hide();
                });
              });
        </script>
    </xsl:template>     
</xsl:stylesheet>