/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls a ribbon button representing the JCR session status (pending changes or not).
 */
Ext.define('Ametys.repository.controller.SessionController',
{
    extend: 'Ametys.repository.controller.RepositoryButtonController',
    
    constructor: function()
    {
        this.callParent(arguments);
        
        this.disable();
        
        Ametys.message.MessageBus.on(Ametys.message.Message.SELECTION_CHANGED, this._onMessageSelectionChanged, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.CREATED, this._onModification, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFYING, this._onModification, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onSaveOrRevertChanges, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.REVERTED, this._onSaveOrRevertChanges, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.DELETED, this._onModification, this);
    },
    
    /**
     * Called when the selection has changed.
     * @param {Ametys.message.Message} message The bus message.
     * @protected
     */
    _onMessageSelectionChanged: function(message)
    {
        var target = message.getTarget(Ametys.message.Message.REPOSITORY_NODE);
        if (target != null)
        {
            var workspaceName = target.getParameters().workspaceName;
            
            // TODO Cache the status by workspace to avoid making a server request every time the selection changes?
            this.serverCall('hasPendingChanges', [workspaceName], this._hasPendingChangesCb, { errorMessage: true, refreshing: true });
        }
    },
    
    /**
     * Enable or disable the buttons depending on the session status.
     * @param {Boolean} hasPendingChanges `true` if the session has pending changes.
     * @protected
     */
    _hasPendingChangesCb: function(hasPendingChanges)
    {
        this.setDisabled(hasPendingChanges !== true);
    },
    
    /**
     * This listener is invoked when a modification has been done on a node (creation, deletion, modification)
     * @param {Ametys.message.Message} message The bus message.
     * @protected
     */
    _onModification: function(message)
    {
        var target = message.getTarget(Ametys.message.Message.REPOSITORY_NODE);
        if (target != null)
        {
            // A node has been created, modified or deleted: enable the controller.
            this.enable();
        }
    },
    
    /**
     * This listener is invoked when the session was saved or rolled back.
     * @param {Ametys.message.Message} message The bus message.
     * @protected
     */
    _onSaveOrRevertChanges: function(message)
    {
        var target = message.getTarget(Ametys.message.Message.REPOSITORY_SESSION);
        if (target != null)
        {
            // The session has been saved or rolled back: disable the controller.
            this.disable();
        }
    }

});
