/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/**
 * Provides a widget for text field<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_STRING.<br>
 * It does handle multiple values (see {@link #cfg-multiple}) using separated by commas text. 
 */
Ext.define('Ametys.form.widget.Text', {
    extend: "Ext.form.field.Text",
    
    /**
     * @cfg {Boolean} multiple=false True to handle multiple values 
     */

    constructor: function (config)
    {
        var validationConfig = config.validationConfig || {},
            maxLength = config.maxLength || validationConfig.maxlength,
        
        config = Ext.apply(config, {
            ametysShowMultipleHint: config.multiple,
            maxLength: maxLength ? Number(maxLength) : undefined
        });
        
        this.callParent(arguments);
    },
    
    getValue: function ()
    {
        var me = this,
        val = me.callParent(arguments);
        
        if (this.multiple)
        {
            var values = val ? val.split(',') : [];
            for (var i = 0; i < values.length; i++)
            {
                values[i] = values[i].trim();
            }
            
            return values;
        }
        
        return val;
    },
    
    getSubmitValue: function()
    {
        return this.multiple ? Ext.encode(this.getValue()) : this.getValue();
    }
});

/**
 * Provides a widget for string enumeration, ordered by alphabetical order.<br>
 * This widget is the default widget registered for enumerated field of type Ametys.form.WidgetManager#TYPE_STRING.<br>
 * It does NOT handle multiple values.
 */
Ext.define('Ametys.form.widget.ComboBox', {
    extend: "Ext.form.field.ComboBox",
    
    /**
     * @cfg {Boolean} naturalOrder=false True to sort drop down list by natural order. By default alphabetical order is applied to the store.
     */
    
    /**
     * @cfg {Boolean} anyMatch=true True to allow matching of the typed characters at any position in the valueField's value.
     */

    constructor: function (config)
    {
        config.valueField = config.valueField || 'value';
        config.displayField = config.displayField || 'text';
            
        var storeCfg = this.getStoreCfg(config),
            typeAhead, editable, anyMatch;
        
        if (Ext.isBoolean(config.typeAhead))
        {
            typeAhead = config.typeAhead; 
        }
        else if (config.typeAhead == "true" || config.typeAhead == "false")
        {
            typeAhead = config.typeAhead == "true";
        }
        
        if (Ext.isBoolean(config.editable))
        {
            editable = config.editable; 
        }
        else if (config.editable == "true" || config.editable == "false")
        {
            editable = config.editable == "true";
        }
        
        config.anyMatch = Ext.isBoolean(config.anyMatch) ? config.anyMatch : config.anyMatch != "false"; // default to true
        
        config = Ext.apply(config, {
            typeAhead: typeAhead !== undefined ? typeAhead : (config.multiple ? false : true),
            editable: editable !== undefined ? editable : (config.multiple ? false : true),
            forceSelection: true,
            triggerAction: 'all',
            
            store: storeCfg,
            multiSelect: config.multiple
        });
        
        this.callParent(arguments);
        
        // Trigger drop down when clicking the field even when editable is false
        if (!config.multiple)
        {
            this.on('click', Ext.bind(this._onClick, this), this, {element: 'inputEl'});
        }
    },
    
    /**
     * Get the store configuration
     * @param {Object} config The current configuration object
     * @return {Object} The store configuration
     */
    getStoreCfg: function(config)
    {
        var storeCfg = {
            id: 0,
            fields: [ config.valueField, {name: config.displayField, type: 'string'}]
        };
        
        if (config.enumeration)
        {
            storeCfg = Ext.apply(storeCfg, {
                type: 'array', // Ext.data.ArrayStore
                data: config.enumeration
            });

            config.queryMode = 'local';
        }
        else if (config.proxyStore)
        {
            storeCfg = Ext.apply(storeCfg, {
                type: 'store', // Ext.data.Store
                proxy: config.proxyStore,
                autoLoad: config.autoLoad || false
            });
            
            config.queryMode = 'remote';
        }
        else
        {
            this.getLogger().error("The combobox widget is not well configured. Should have enumeration or proxy store.");
        }
        
        config.naturalOrder = Ext.isBoolean(config.naturalOrder) ? config.naturalOrder : config.naturalOrder == 'true';
        if (!config.naturalOrder)
        {
            storeCfg.sorters = [{property: config.displayField, direction:'ASC'}]; // default order
        }
        
        return storeCfg;
    },
    
    /**
     * @private
     * Function invoked when the combo box is clicked
     */
    _onClick: function()
    {
        // Do not collapse when the combo box is already expanded
        if (!this.readOnly && !this.disabled && !this.isExpanded) 
        {
            this.onTriggerClick();
        }
    }
});

/**
 * Provides a widget for long input field.<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_LONG.<br> 
 * It does NOT handle multiple values.
 */
Ext.define('Ametys.form.widget.Long', {
    extend: "Ext.form.field.Number",
    
    constructor: function (config)
    {
        config = Ext.apply(config, {
            allowDecimals: false
        });
        
        this.callParent(arguments);
    }
});

/**
 * Provides a widget for multiple long input field.<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_LONG.<br>
 * It only handle multiple values.
 */
Ext.define('Ametys.form.widget.LongMulti', {
    extend: "Ext.form.field.TextArea",

    statics:
    {
        /**
         * @property {Number} FIELD_HEIGHT The default height for textarea field
         * @private
         * @readonly
         */
        FIELD_HEIGHT: 80
    },

    /**
     * @cfg {String} errorText
     * The error text to display if the validation fails
     */
    errorText : "{{i18n PLUGINS_CORE_UI_WIDGET_LONG_MULTI_FIELD_ERROR}}",

    /**
     * Constructor of the widget to handle multiple long values
     * @param {Object} config widget config
     */
    constructor: function (config)
    {
        var height = config.height;

        config = Ext.apply(config, {
            height: height ? Number(height) : Ametys.form.widget.TextArea.FIELD_HEIGHT
        });

        this.callParent(arguments);
    },

    /**
     * Sets a long value into the field
     * @param {Number[]} value The value to set as an array of number
     */
    setValue: function(value)
    {
        if (!value)
        {
            this.callParent(arguments);
        }
        else
        {
            if (Ext.isArray(value))
            {
                value = value.join("\n");
            }

            this.callParent([value]);
        }
    },

    getValue: function()
    {
        var value = this.callParent(arguments) || '';

        if (value)
        {
            var strValues = value.split(/\r?\n/);
            
            var me = this,
                parseValues = [];
            Ext.Array.each(strValues, function (strValue) {
                strValue = strValue.trim();
                if (!Ext.isEmpty(strValue))
                {
                    var parseValue = me.parseValue(strValue);
                    if (parseValue != null)
                    {
                        parseValues.push(parseValue);
                    }
                }
            });
            
            return parseValues;
        }

        return [];
    },

    getErrors: function (value)
    {
        value = value || this.getRawValue();// Just in case
        var errors = this.callParent(arguments);

        if (value)
        {
            var strValues = value.split(/\r?\n/);
            
            var me = this,
                invalidValues = [];
            Ext.Array.each(strValues, function (strValue) {
                strValue = strValue.trim();
                if (!Ext.isEmpty(strValue))
                {
                    var intValue = me.parseValue(strValue);
                    if (intValue == null)
                    {
                        invalidValues.push(strValue);
                    }
                }
            });
            
            if (invalidValues.length > 0)
            {
                errors.push(this.errorText + invalidValues.join(", "));
            }
        }

        return errors;
    },

    /**
     * Parse a single String value to a Number
     * @param {String} value
     * @return {Number}
     * @protected
     */
    parseValue : function(value)
    {
        var intValue = parseInt(value);
        if (isNaN(intValue) || "" + intValue != value.trim())
        {
            this.getLogger().warn(value + " is not a number");
            return null;
        }
        else
        {
            return intValue;
        }
    },

    getSubmitValue: function ()
    {
        return !this.value ? null : Ext.JSON.encode(this.getValue());
    }
});

/**
 * Provides a widget for decimal input field.<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_DOUBLE.<br>
 * It does NOT handle multiple values.
 */
Ext.define('Ametys.form.widget.Double', {
    extend: "Ext.form.field.Number",

    constructor: function (config)
    {
        config = Ext.apply(config, {
            allowDecimals: true
        });
        
        this.callParent(arguments);
    }
});

/**
 * Provides a widget for multiple decimal input field.<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_DOUBLE.<br>
 * It only handle multiple values.
 */
Ext.define('Ametys.form.widget.DoubleMulti', {
    extend: "Ametys.form.widget.LongMulti",

    /**
     * @cfg {String} decimalSeparator
     * Character(s) to allow as the decimal separator. Defaults to {@link Ext.util.Format#decimalSeparator decimalSeparator}.
     * @locale
     */
    decimalSeparator : null,

    /**
     * @cfg {String} errorText
     * The error text to display if the validation fails
     */
    errorText : "{{i18n PLUGINS_CORE_UI_WIDGET_DOUBLE_MULTI_FIELD_ERROR}}",

    initComponent: function() {
        var me = this;
        if (me.decimalSeparator === null) {
            me.decimalSeparator = Ext.util.Format.decimalSeparator;
        }
        me.callParent();
    },

    /**
     * Sets a decimal value into the field
     * @param {Number[]} value The value to set as an array of number
     */
    setValue: function(value)
    {
        if (!value)
        {
            this.callParent(arguments);
        }
        else
        {
            if (Ext.isArray(value))
            {
                value = value.join("\n");
                value = value.replace(/\./g, this.decimalSeparator);
            }

            this.callParent([value]);
        }
    },

    /**
     * Parse a single String value to a Number
     * @param {String} value
     * @return {Number}
     * @protected
     */
    parseValue : function(value)
    {
        var translatedValue = String(value).replace(this.decimalSeparator, '.').trim();
        var floatValue = parseFloat(translatedValue);

        // Tests that the double is the same value as inputed (to avoid that 12AAAAA is parsed as 12 by javascript)
        var test = translatedValue + "1";// Ugly test in case the user input something like 12.00 (the double will be 12)
        var testValue = parseFloat(test);

        if (isNaN(floatValue) || "" + testValue != test)
        {
            this.getLogger().warn(value + " is not a number");
            return null;
        }
        else
        {
            return floatValue;
        }
    }
});

/**
 * Provides a widget for checkbox input field.<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_BOOLEAN.<br> 
 * It does NOT handle multiple values.
 */
Ext.define('Ametys.form.widget.Checkbox', {
    extend: "Ext.form.field.Checkbox",
    
    constructor: function (config)
    {
        config = Ext.apply(config, {
            checked: Ext.isBoolean(config.value) ? config.value : config.value == "true",
            hideEmptyLabel: Ext.isBoolean(config.hideEmptyLabel) ? config.hideEmptyLabel : config.hideEmptyLabel !== "false",   
            hideLabel: Ext.isBoolean(config.hideLabel) ? config.hideLabel : config.hideLabel == "true",     
            inputValue: 'true', 
            uncheckedValue: 'false'
        });
        
        this.callParent(arguments);
    }
});

/**
 * Provides a widget for date input field.<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_DATE.<br> 
 * It does NOT handle multiple values.<br>
 * 
 * The format used for date is ISO 8601
 */
Ext.define('Ametys.form.widget.Date', {
    extend: "Ext.form.field.Date",
    
    /**
     * @cfg {Boolean} initWithCurrentDate=false True to initialize the field with the current date.
     */
        
    constructor: function (config)
    {
        config = Ext.apply(config, {
            value: this._getDateValue(config),
            format: Ext.Date.patterns.LongDate,
            altFormats: 'c',
            submitFormat: Ext.Date.patterns.ISO8601DateTime
        });

        this.callParent(arguments);
    },
    
    /**
     * @private
     * Function invoked when the combo box is clicked
     */
    _getDateValue: function(config)
    {
        var initWithCurrentDate = Ext.isBoolean(config.initWithCurrentDate) ? config.initWithCurrentDate : config.initWithCurrentDate === "true";
        var value = config.value;
        if (!config.value && initWithCurrentDate)
        {
            value = new Date();
        }
        return value;
    }
});

/**
 * Provides a widget for datetime input field.<br>
 * This widget is available for fields of type Ametys.form.WidgetManager#TYPE_DATE.<br> 
 * It does NOT handle multiple values.<br>
 * 
 * The format used for date is ISO 8601
 */
Ext.define('Ametys.form.widget.DateTime', {
    extend: "Ametys.form.field.DateTime",
    
    /**
     * @cfg {Boolean} initWithCurrentDate=false True to initialize the field with the current date.
     */
        
    constructor: function (config)
    {
        config.dateConfig = Ext.apply(config.dateConfig || {}, {
            value: this._getDateValue(config),
            format: Ext.Date.patterns.LongDate,
            altFormats: 'c',
            submitFormat: Ext.Date.patterns.ISO8601Date
        });
        
        config.timeConfig = Ext.apply(config.timeConfig || {}, {
            value: this._getDateValue(config),
            format: "H:i"
        });
        
        delete config.value;
        
        this.callParent(arguments);
    },
    
    /**
     * @private
     * Function invoked when the combo box is clicked
     */
    _getDateValue: function(config)
    {
        var initWithCurrentDate = Ext.isBoolean(config.initWithCurrentDate) ? config.initWithCurrentDate : config.initWithCurrentDate === "true";
        var value = config.value;
        if (!config.value && initWithCurrentDate)
        {
            value = new Date();
        }
        return value;
    }
});

/**
 * Provides a widget for rich text field.<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_RICH_TEXT.<br>
 * It does NOT handle multiple values.<br>
 * It can be "remotely" configured by using {@link Ametys.form.widget.RichText.RichTextConfiguration}
 */
Ext.define('Ametys.cms.form.widget.RichText', {
    extend: "Ext.form.field.RichText",
    
    statics:
    {
        /**
         * @property {Number} FIELD_HEIGHT The default height for textarea field
         * @private
         * @readonly 
         */
        FIELD_HEIGHT: 400
    },
    
    constructor: function (config)
    {
        var validationConfig = config.validationConfig || {},
            maxLength = config.maxLength || validationConfig.maxlength,
            height = config.height;
        
        config = Ext.apply(config, {
            height: height ? Number(height) : Ametys.cms.form.widget.RichText.FIELD_HEIGHT,
            maxLength: maxLength ? Number(maxLength) : undefined,
            resizable: true,
            charCounter: true,
            checkTitleHierarchy: true,
            
            editorCSSFile: Ametys.form.widget.RichText.RichTextConfiguration.getCSSFiles(),
            validElements: config.validElements || Ametys.form.widget.RichText.RichTextConfiguration.getTags(),
            validStyles: config.validStyles ? Ext.JSON.decode(config.validStyles) : Ametys.form.widget.RichText.RichTextConfiguration.getStylesForTags(),
            validClasses: config.validClasses ? Ext.JSON.decode(config.validClasses) : Ametys.form.widget.RichText.RichTextConfiguration.getClassesForTags(),
            
            validator: function(value)
            {
                return Ametys.form.widget.RichText.RichTextConfiguration.validates(value);
            }
        });
        
        this.callParent(arguments);
        
        this.addListener('editorsetcontent', function(field, editor, object) { Ametys.form.widget.RichText.RichTextConfiguration.convertOnSetContent(field, editor, object); });
        this.addListener('editorgetcontent', function(field, editor, object) { Ametys.form.widget.RichText.RichTextConfiguration.convertOnGetContent(field, editor, object); });
        
        this.addListener('editorkeypress', function(field, editor, e) { Ametys.form.widget.RichText.RichTextConfiguration.fireEvent('keypress', field, editor, e); });
        this.addListener('editorkeydown', function(field, editor, e) { Ametys.form.widget.RichText.RichTextConfiguration.fireEvent('keydown', field, editor, e); });
        this.addListener('editorkeyup', function(field, editor, e) { Ametys.form.widget.RichText.RichTextConfiguration.fireEvent('keyup', field, editor, e); });
        this.addListener('editorvisualaid', function(field, editor, object) { Ametys.form.widget.RichText.RichTextConfiguration.fireEvent('visualaid', field, editor, object); });
        this.addListener('editorpreprocess', function(field, editor, object) { Ametys.form.widget.RichText.RichTextConfiguration.fireEvent('preprocess', field, editor, object); });
        this.addListener('editorhtmlnodeselected', function(field, editor, node) { Ametys.form.widget.RichText.RichTextConfiguration.fireEvent('htmlnodeselected', field, editor, node); });
    }
});

/**
 * Provides a widget for textarea field (multiline text).<br>
 * This widget is registered for fields of type Ametys.form.WidgetManager#TYPE_STRING.<br>
 * It does NOT handle multiple values. 
 */
Ext.define('Ametys.form.widget.TextArea', {
    extend: "Ametys.form.field.TextArea",
    
    statics:
    {
        /**
         * @property {Number} FIELD_HEIGHT The default height for textarea field
         * @private
         * @readonly 
         */
        FIELD_HEIGHT: 80
    },
        
    constructor: function (config)
    {
        var validationConfig = config.validationConfig || {},
            maxLength = config.maxLength || validationConfig.maxlength,
            height = config.height;
        
        config = Ext.apply(config, {
            height: height ? Number(height) : Ametys.form.widget.TextArea.FIELD_HEIGHT,
            maxLength: maxLength ? Number(maxLength) : null,
            charCounter: Ext.isBoolean(config.charCounter) ? config.charCounter : config.charCounter == "true",
        });
        
        this.callParent(arguments);
    }
    
});

/**
 * Provides a widget for reference of type external<br>
 * This widget is the default widget registered for fields of type Ametys.form.WidgetManager#TYPE_REFERENCE.<br>
 * It does NOT handle multiple values.
 */
Ext.define('Ametys.form.widget.UrlReference', {
    extend: 'Ametys.form.widget.Text',
    
    /**
     * @property {String} referenceType=__external The type of the reference.
     * @readonly 
     */
    referenceType: '__external',
    
    /**
     * Sets a data value into the field 
     * @param {Object|String} value The value to set as a Object or a JSON encoded object
     * @param {Number} value.referenceType The type of reference
     * @param {Number} value.value The url
     */
    setValue: function(value) 
    {
        if (!value)
        {
            this.callParent(arguments);
        }
        else 
        {
            if (!Ext.isObject(value))
            {
                try
                {
                    value = Ext.JSON.decode(value);
                }
                catch(e)
                {
                    value = {
                        type: this.referenceType,
                        value: value
                    }
                }
            }
            
            this.callParent([value.value]);
        }
    },
    
    /**
     * @inheritdoc
     */
    getValue: function()
    {
        var value = this.callParent(arguments) || '';
        
        if (value)
        {
            return {
                type: this.referenceType,
                value: value
            }
        }
        
        return value;
    },
    
    getSubmitValue: function ()
    {
        return !this.value ? null : Ext.JSON.encode(this.getValue());
    }
});
