/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton to manage the 'edition.richtext' widget for the FormEditionPanel.
 * Allow to add CSS files, supported tags and common listeners.
 */
Ext.define('Ametys.form.widget.RichText.RichTextConfiguration', {
	extend: 'Ext.util.Observable',
	
	singleton: true,
	
	/**
	 * @property {Object} _useCSSFile A map of category / css files to load. "" is the default category
     * @property {String[]} _useCSSFile.key List of css files to load in the inline editor for a category
	 * @private
	 */
	_useCSSFile: {},
	/**
	 * @property {Object} _tags A map of category / handled tags. "" is the default category.
     * @property {Object} _tags.key Handled tags for a category
	 * @private
	 */
	_tags: {},
    /**
     * @property {Object} __styles A map of category / handled styles. "" is the default category.
     * @property {Object} __styles.key Handled styles for a category
     * @private
     */
    _styles: {},
    
	/**
	 * @property {Object} _validators A map of cagtegory / list of registered validator. "" is the default category
     * @property {Function[]} _validators.key List of registered validator #addValidator used in #validates for a category
	 * @private
	 */
	_validators: {},
    /**
     * @property {Object} __convertorsOnGetContent A map of category / list of registered convertors. "" is the default category.
     * @property {Function[]} __convertorsOnGetContent.key List of registered convertors #addConvertors used in #convertOnGetContent for a category
     * @private
     */
    _convertorsOnGetContent: {},
    /**
     * @property {Object} __convertorsOnSetContent A map of category / list of registered convertors. "" is the default category
     * @property {Function[]} __convertorsOnSetContent.key List of registered convertors #addConvertors used in #convertOnSetContent for a category
     * @private
     */
    _convertorsOnSetContent: {},
    
    /**
     * @private
     * @property {Boolean} _initialized Is the RichTextConfiguration ready to be used? 1 means yes, and a negative file means files are beeing loaded
     */
    _initialized: false,
	
	/**
     * @event setcontent
     * Fires when the editor received new content. This allows to convert storing tags to internal tags. Use object.content to get/set the full html. See Ametys.form.field.RichText#event-setcontent for parameters.
     */
    /**
     * @event getcontent
     * Fires when the editor received content. This allows to convert internal tags to storing tags. Use object.content to get/set the full html. See Ametys.form.field.RichText#event-getcontent for parameters.
     */
    /**
     * @event keypress
     * Fires when the editor has a key press. See Ametys.form.field.RichText#event-keypress for parameters.
     */
    /**
     * @event keydown
     * Fires when the editor has a key down. See Ametys.form.field.RichText#event-keydown for parameters.
     */
    /**
     * @event keyup
     * Fires when the editor has a key up. See Ametys.form.field.RichText#event-keydown for parameters.
     */
    /**
     * @event visualaid
     * Fires when the editor pre process the serialization. See Ametys.form.field.RichText#event-visualaid for parameters.
     */
    /**
     * @event preprocess
     * Fires when the editor pre process the serialization. See Ametys.form.field.RichText#event-preprocess for parameters.
     */
    /**
     * @event htmlnodeselected
     * Fires when a HTML node is selected in editor. See Ametys.form.field.RichText#event-htmlnodeselected for parameters.
     */
    
    constructor: function()
    {
        this.callParent(arguments);

        Ametys.data.ServerComm.callMethod({
            role: "org.ametys.core.ui.widgets.richtext.RichTextConfigurationExtensionPoint",
            methodName: "toJSON",
            parameters: [ Ametys.getAppParameters() ],
            callback: {
                handler: this._setConfigurationFromServer,
                scope: this
            },
            errorMessage: true
        });
    },
    
    /**
     * @private
     * Called as a callback of the RichTextConfigurationExtensionPoint.toJSON method
     * @param {Object} json The json representing the configuration required for the richtexts. See the javadoc of RichTextConfigurationExtensionPoint.toJSON to know the format.
     */
    _setConfigurationFromServer: function(json)
    {
        function getFilesUrl(files)
        {
            var urls = [];
            
            Ext.Array.forEach(files, function (file) {
                var url;
                
                if (file.language)
                {
                    url = file.url[Ametys.LANGUAGE_CODE] || file.url[file['default'] || ''];
                }
                else
                {
                    if ((file.debug == null || file.debug == "all" || Ametys.DEBUG_MODE + "" == file.debug)
                        && (file.rtl == null || file.rtl == "all" || Ametys.RTL_MODE + "" == file.rtl))
                    {
                        url = file.url;
                    }
                }
                
                if (url)
                {
                    urls.push(Ametys.CONTEXT_PATH + url);
                }
            });

            
            return urls;
        }
        
        var me = this;
        if (json)
        {
            Ext.Object.each(json, function(category, jsonCategory) {
                if (jsonCategory.tags)
                {
                    Ext.Object.each(jsonCategory.tags, function(tagName, jsonTag) {
                        var tag = me.handleTag(tagName, category);
                        
                        if (jsonTag.empty)
                        {
                            switch (jsonTag.empty)
                            {
                                default:
                                case "CLOSE": tag.emptyTag = ""; break;
                                case "OPEN": tag.emptyTag = "+"; break;
                                case "REMOVE_EMPTY_CONTENT": tag.emptyTag = "-"; break;
                                case "PADDING": tag.emptyTag = "#"; break;
                                case "REMOVE_EMPTY_ATTRIBUTES": tag.emptyTag = "!"; break;
                            }
                        }
                        
                        if (jsonTag.synonyms)
                        {
                            Ext.Array.forEach(jsonTag.synonyms, function(synonym) {
                                tag.replaceTags.push(synonym);
                            })
                        }
                        
                        if (jsonTag.attributes)
                        {
                            Ext.Object.each(jsonTag.attributes, function(attributeName, jsonAttribute) {
                                var attribute = tag.handleAttribute(attributeName);
                                
                                attribute.defaultValue = jsonAttribute["default-value"];
                                attribute.handleValue(jsonAttribute["values"]);
                                if (jsonAttribute["technical-values"])
                                {
                                    attribute.handleTechnicalValue(jsonAttribute["technical-values"]);
                                }
                            });
                        }
                    });
                }
                
                if (jsonCategory.styles)
                {
                    Ext.Object.each(jsonCategory.styles, function(type, jsonStyles) {
                        Ext.Array.each(jsonStyles.groups, function(jsonGroup) {
                            var group = me.handleStyledGroup(type, jsonGroup.label, jsonGroup.priority, category);
                            
                            Ext.Array.each(jsonGroup.values, function(jsonStyle) {
                                group.handleStyledElements(jsonStyle.tagname, jsonStyle.cssclass, jsonStyle.label, jsonStyle.description, jsonStyle.buttonCSSClass, jsonStyle.buttonSmallIcon, jsonStyle.buttonMediumIcon, jsonStyle.buttonLargeIcon);
                            });
                        });
                        
                    });
                }
                
                if (jsonCategory.css)
                {
                    Ext.Array.forEach(getFilesUrl(jsonCategory.css), function (cssFile) {
                        me.addCSSFile(cssFile);
                    });
                }
                
                if (jsonCategory.validators)
                {
                    Ext.Array.forEach(jsonCategory.validators, function(validator) {
                        var validatorInstance = Ext.create(validator['class'].name, validator['class'].parameters);
                        me.addValidator(Ext.bind(validatorInstance.validates, validatorInstance), category);
                    });
                }
                
                if (jsonCategory.convertors)
                {
                    Ext.Array.forEach(jsonCategory.convertors, function(convertor) {
                        var convertorInstance = Ext.create(convertor['class'].name, convertor['class'].parameters);
                        me.addConvertor(convertorInstance.onGetContent ? Ext.bind(convertorInstance.onGetContent, convertorInstance) : null, convertorInstance.onSetContent ? Ext.bind(convertorInstance.onSetContent, convertorInstance) : null, category);
                    });
                }
            });
            
            this._initialized = true;
        }
    },
	
    /**
     * @private
     * Add a custom validation function to be called during inline editor validation ({@link Ext.form.field.Field#getErrors}).
     * @param {Function} validator The new validator to add. This function will have the following signature:
     * @param {Object} validator.value The current field value
     * @param {Boolean/String} validator.return
     * - True if the value is valid
     * - An error message if the value is invalid
     * @param {String} [category=""] The category where to register.
     */
	addValidator: function(validator, category)
	{
        category = category || "";
        
        this._validators[category] = this._validators[category] || []; 
		this._validators[category].push(validator);
	},
	
    /**
     * Validates the value among the existing validators for the inline editor
     * @param {String} value The current field value
     * @param {String} [category=""] The category to validates
     * @return {Boolean/String} validator.return
     *
     * - True if the value is valid
     * - An error message if the value is invalid
     */
	validates: function(value, category)
	{
        this.checkIfInitialized();

        category = category || "";
        
		var returnValues = "";
		
		Ext.each(this._validators[category] || [], function (validator) {
			var returnValue = validator.apply(null, [value]);
			if (returnValue !== true)
			{
				returnValues += returnValue + "\n";
			}
		});
		
		return returnValues.length == 0 ? true : returnValues.substring(0, returnValues.length - 1);
	},
	
    /**
     * @private
     * Add a custom convertor function to be called during inline editor conversion process getValue/setValue.
     * @param {Function} onGetContent The new function to add that will be called on getValue to convert the internal richtext structure to the external value
     * @param {Ametys.form.field.RichText} onGetContent.field The current richtext field
     * @param {tinymce.Editor} onGetContent.editor The current richtext editor
     * @param {Object} onGetContent.object The object of value to be modified
     * @param {Function} onSetContent The new function to add that will be called on setValue to convert the external value to the internal richtext structure 
     * @param {tinymce.Editor} onSetContent.editor The current richtext editor
     * @param {Object} onSetContent.object The object of value to be modified
     * @param {String} [category=""] The category where to register.
     */
    addConvertor: function(onGetContent, onSetContent, category)
    {
        category = category || "";
        
        if (onGetContent)
        {
            this._convertorsOnGetContent[category] = this._convertorsOnGetContent[category] || []; 
            this._convertorsOnGetContent[category].push(onGetContent);
        }

        if (onSetContent)
        {
            this._convertorsOnSetContent[category] = this._convertorsOnSetContent[category] || []; 
            this._convertorsOnSetContent[category].push(onSetContent);
        }
    },
    
    /**
     * Converts the value on get content
     * @param {Ametys.form.field.RichText} field The richtext to convert
     * @param {tinymce.Editor} editor The current richtext editor
     * @param {Object} object The object of value to be modified
     * @param {String} [category=""] The category to apply.
     */
    convertOnGetContent: function(field, editor, object, category)
    {
        this.checkIfInitialized();

        category = category || "";

        Ext.Array.each(this._convertorsOnGetContent[category] || [], function (convertorOnGetContent) {
            convertorOnGetContent.apply(null, [field, editor, object]);
        });
        
        this.fireEvent('getcontent', field, editor, object);
    },
    
    /**
     * Converts the value on set content
     * @param {Ametys.form.field.RichText} field The richtext to convert
     * @param {tinymce.Editor} editor The current richtext editor
     * @param {Object} object The object of value to be modified
     * @param {String} [category=""] The category to apply.
     */
    convertOnSetContent: function(field, editor, object, category)
    {
        this.checkIfInitialized();

        category = category || "";

        Ext.Array.each(this._convertorsOnSetContent[category] || [], function (convertorOnSetContent) {
            convertorOnSetContent.apply(null, [field, editor, object]);
        });
        
        this.fireEvent('setcontent', field, editor, object);
    },
    
	/**
     * @private
	 * Add a CSS file to load in the inline editor
	 * @param {String} file The path of CSS file 
     * @param {String} [category=""] The category to register.
	 */
	addCSSFile: function (file, category)
	{
        category = category || "";
        this._useCSSFile[category] = this._useCSSFile[category] || [];
		this._useCSSFile[category].push(file);
	},
	
	/**
	 * Get all added css files as one string
     * @param {String} [category=""] The category to apply.
	 * @return {String} The comma-separated list of added files
	 */
	getCSSFiles: function(category)
	{
        this.checkIfInitialized();

        category = category || "";
		return (this._useCSSFile[category] || []).join(",")
	},
	
	/**
	 * Get all supported tags as a tinymce string. See valid_element tinymce configuration doc for the exact format.
     * @param {String} [category=""] The category to apply.
	 * @return {String} The supported tags.
	 */
	getTags: function(category)
	{
        this.checkIfInitialized();

        category = category || "";
        var tags = this._tags[category] || {};

        var validElements = "";
		for (var key in tags)
		{
			if (validElements != "")
			{
				validElements += ",";
			}
			validElements += tags[key].toString();
		}
		return validElements;
	},
    
    /**
     * Get all supported styles as a tinymce conf object. See valid_styles tinymce configuration doc for the exact format.
     * @param {String} [category=""] The category to apply.
     * @return {Object} The supported properties for the style attribute.
     */
    getStylesForTags: function(category)
    {
        this.checkIfInitialized();
        
        category = category || "";
        var tags = this._tags[category] || {};
        
        var validStyles = {};
        
        for (var key in tags)
        {
            if (tags[key].attributes["style"])
            {
                validStyles[key] = tags[key].attributes["style"].values.join(",");
            }
        }        
        
        return validStyles;
    },
    
    /**
     * Get all supported classes as a tinymce conf object. See valid_classes tinymce configuration doc for the exact format.
     * @param {String} [category=""] The category to apply.
     * @return {Object} The supported properties for the style attribute.
     */
    getClassesForTags: function(category)
    {
        this.checkIfInitialized();

        category = category || "";
        var tags = this._tags[category] || {};
        
        var validClasses = {};
        
        for (var key in tags)
        {
            validClasses[key] = {};
            
            if (tags[key].attributes["class"])
            {
                validClasses[key] = tags[key].attributes["class"].values.join(" ");
            }
        }        
        
        return validClasses;
    },    
	
	/**
     * @private
	 * This method retrieve a tag to handle.
	 * Just by calling this method, the tag will be handled by the editor.
	 * Once you have a tag it a map
	 *  * emptyTag Can be empty, +, -, # or !. see tinymce valid_elements documentation
	 *  * getAttributes Call this method to handle an attribute. This will return a map where key is the attribute name and value are:
	 *     * values An array of possible values
	 *     * defaultValue The default value. Can be null if there is no
	 *  * replaceTags An array of tags that will be replaced by this one. This value can be dynamically removed if some one register this other tag as handled.
	 * 
	 * e.g. 
     * <code>
     *      var aTag = Ametys.form.widget.RichText.RichTextConfiguration.handleTag("a"); // from now &lt;a&gt; will be accepted
	 *      aTag.emptyTag = "+"; // The &lt;a&gt; tag will be forced opened
	 *      var classAttr = aTag.handleAttribute("class"); // The class attribute is now accepted on the &lt;a&gt; tag
	 *      classAttr.defaultValue = "myclass"; // The class attribute will now always exists on &lt;a&gt; tag with the value "myclass"
	 *      classAttr.handleValue("myclass");
	 *      classAttr.handleValue("myclass2"); // The class attribute will now accept 2 values myclass or myclass2
     *      classAttr.handleValue(["myclass3", "myclass"]); // The class attribute will now accept 3 values myclass or myclass2 or myclass3
     *      classAttr.handleTechnicalValue("floatright"); // The class attribute will accept this new value and moreover this value is tagged as 'technical' for some ApplyStyle code
     * </code>
     *      
     * Please note, that the "style" and "classes" attributes are holded separately:
     *  * For "style", each value is a valid property. E.g. the following line wille handle the "p" tag, with a style attribute, which contains the 'text-align' property.
     *      Ametys.form.widget.RichText.RichTextConfiguration.handleTag("p").handleAttribute("style").handleValue("text-align");
     *      and not a style attribute that can be equals to "text-align".
     *  * For "class", each value is a valid class.
     *      Ametys.form.widget.RichText.RichTextConfiguration.handleTag("p").handleAttribute("class").handleValue("a");
     *      means that the class attribute can contains "a", but do not need to be equals to "a".
     * @param {String} tagName The name of the tag to handle in the configuration
     * @param {String} [category=""] The category of configuration where to handle
     * @return {Object} The tag objet to handle attributes, empty behavior... See description above. 
	 */
	handleTag: function(tagName, category)
	{
        category = category || "";
        this._tags[category] = this._tags[category] || {};
        
		if (this._tags[category][tagName] == null)
		{
			var me = this;
			this._tags[category][tagName] = {
					tagName: tagName,
					emptyTag: "",
					attributes: {},
                    getAttribute: function(attributeName)
                    {
                        return this.attributes[attributeName];
                    },
					handleAttribute: function(attributeName)
					{
						if (this.attributes[attributeName] == null)
						{
							this.attributes[attributeName] = {
								attributeName: attributeName,
								values: [],
                                technicalValues: [],
								defaultValue: null,
                                handleValue: function(value) {
                                    value = Ext.Array.from(value);
                                    this.values = Ext.Array.merge(this.values, value)
                                    return this;
                                },
                                handleTechnicalValue: function(value) {
                                    value = Ext.Array.from(value);
                                    this.values = Ext.Array.merge(this.values, value)
                                    this.technicalValues = Ext.Array.merge(this.technicalValues, value)
                                    return this;
                                },
								toString: function() {
									var a = this.defaultValue != null ? this.attributeName + "=" + this.defaultValue : "";

									var b = ""
                                    if (attributeName != "style" && attributeName != "class")
                                    {
    									for (var key in this.values)
    									{
    										if (typeof this.values[key] == "string")
    										{
    											if (b != "") 
    											{
    												b += "?";
    											}
    																			
    											b += this.values[key];
    										}
    									}
    									if (b != "")
    									{
    										b = this.attributeName + "<" + b;
    									}
                                    }
								
									if (a == "" && b == "")
									{
										return this.attributeName;
									}
									else 
									{
										return a + ((a != "" && b != "") ? "|" : "") + b;
									}
								}
							};
						}
						return this.attributes[attributeName];
					},
					replaceTags: [],
					toString: function() {
						var base = this.emptyTag + tagName;
						
						var attributes = "";
						for (var key in this.attributes)
						{
							if (attributes != "")
							{
								attributes += "|"; 
							}
							attributes += this.attributes[key].toString(); 
						}
						
						var attrs = (attributes == "" ? "" : "[" + attributes + "]");
						
						if (this.replaceTags.length == 0)
						{
							return base + attrs;
						}
						else
						{
							var finals = "";
							for (var i = 0; i < this.replaceTags.length; i++)
							{
								// The second part of the condition allow to remove multiple instance of the same string
								if (me._tags[category][this.replaceTags[i]] == null && this.replaceTags.indexOf(this.replaceTags[i]) == i)
								{
									finals += base + "/" + this.replaceTags[i] + attrs + ",";
								}
							}
							return finals.substring(0, finals.length - 1); 
						}
						
					}
			};
		}
		return this._tags[category][tagName];
	},
    
    /**
     * Get tag informations. 
     * @param {String} tagName The name of the tag to handle in the configuration
     * @param {String} [category=""] The category of configuration where to handle
     * @return {Object} See #handleTag to know more
     */
    getTag: function(tagName, category)
    {
        category = category || "";
        this._tags[category] = this._tags[category] || {};
        return this._tags[category][tagName];        
    },
    
    /**
     * @private
     * Handle a new group of style
     * @param {String} type The type of element to style. Such as "paragraph", "table", "link", "image", "ol", "ul"...
     * @param {String} label The group label
     * @param {Number} priority The priority of the group. Low is first.
     * @param {String} [category=""] The category of configuration where to handle
     * @return {Object} The styled group
     * @return {String} return.label The group label
     * @return {Number} return.priority The group priority
     * @return {Object[]} return.values The values for this group
     * @return {String} return.values.tagName The specific tagname to use for this style (available for 'paragraph' only)
     * @return {String} return.values.cssClass The classname to use for this style (can be null for a 'paragraph') 
     * @return {String} return.values.label The label of the button to affect the style
     * @return {String} return.values.description The description of the button to affect the style
     * @return {String} return.values.buttonCSSClass A specific CSS class to apply on the button to affect the style 
     * @return {String} return.values.buttonSmallIcon The small image of the button to affect the style
     * @return {String} return.values.buttonMediumIcon The medium image of the button to affect the style
     * @return {String} return.values.buttonLargeIcon The large image of the button to affect the style     
     * @return {Function} return.handleStyledElements Handle a new value 
     * @return {String} return.handleStyledElements.tagName The specific tagname to use for this style (available for 'paragraph' only)
     * @return {String} return.handleStyledElements.cssClass The classname to use for this style (can be null for a 'paragraph') 
     * @return {String} return.handleStyledElements.label The label of the button to affect the style
     * @return {String} return.handleStyledElements.description The description of the button to affect the style
     * @return {String} return.handleStyledElements.buttonCSSClass A specific CSS class to apply on the button to affect the style 
     * @return {String} return.handleStyledElements.buttonSmallIcon The small image of the button to affect the style
     * @return {String} return.handleStyledElements.buttonMediumIcon The medium image of the button to affect the style
     * @return {String} return.handleStyledElements.buttonLargeIcon The large image of the button to affect the style
     */
    handleStyledGroup: function(type, label, priority, category)
    {
        category = category || "";
        this._styles[category] = this._styles[category] || {};
        this._styles[category][type] = this._styles[category][type] || []; 
        
        var newGroup = {
            label: label,
            priority: priority,
            values: [],
            handleStyledElements: function(tagName, cssClass, label, description, buttonCSSClass, buttonSmallIcon, buttonMediumIcon, buttonLargeIcon)
            {
                this.values.push({
                    tagName: tagName,
                    cssClass: cssClass,
                    label: label,
                    description: description,
                    buttonCSSClass: buttonCSSClass,
                    buttonSmallIcon: buttonSmallIcon,
                    buttonMediumIcon: buttonMediumIcon,
                    buttonLargeIcon: buttonLargeIcon
                });
            }
        };
        
        this._styles[category][type].push(newGroup);
        function seg(num)
        {
            if (num > 0) return 1;
            else if (num == 0) return 0;
            else return -1;
        }
        this._styles[category][type].sort(function (g1, g2) { return seg(g1.priority - g2.priority); });
        
        return newGroup;
    },
    
    /**
     * Get styles informations of a type. 
     * @param {String} type The type of element to style. Such as "paragraph", "table", "link", "image", "ol", "ul"...
     * @param {String} [category=""] The category of configuration where to handle
     * @return {Object[]} The styles available by group. See #handleStyledGroup return value. Can be null or empty.
     */
    getStyledElements: function(type, category)
    {
        category = category || "";
        this._styles[category] = this._styles[category] || {};
        return this._styles[category][type];
    },
    
    /**
     * @private
     * Check if RichTextConfiguration is initialized and throw an exception otherwise
     */
    checkIfInitialized: function()
    {
        if (!this._initialized)
        {
            throw "RichTextConfiguration component is not initialized";
        }
    }
});
