/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The workflow function DAO
 */
Ext.define('Ametys.plugins.workflow.dao.WorkflowFunctionDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method addFunction
         * Add workflow function
         * This calls the method 'addFunction' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "addFunction",
                callback: {
                    handler: this._addFunctionCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method editFunction
         * Edit workflow function
         * This calls the method 'editFunction' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "editFunction",
                callback: {
                    handler: this._editFunctionCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method deleteFunction
         * Delete workflow function
         * This calls the method 'deleteFunction' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "deleteFunction",
                callback: {
                    handler: this._deleteFunctionCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method getFunctionsModel
         * Get model items associated with function 
         * This calls the method 'getFunctionsModel' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "getFunctionsModel",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: true,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method getFunctionParametersValues
         * Get argument values associated with function 
         * This calls the method 'getFunctionParametersValues' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "getFunctionParametersValues",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method getFunctionDescription
         * Get description associated with function 
         * This calls the method 'getFunctionDescription' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "getFunctionDescription",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method moveUp
         * Move the function up in the callstack
         * This calls the method 'moveUp' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "moveUp",
                callback: {
                    handler: this._moveUpCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method moveDown
         * Move the function down in the callstack
         * This calls the method 'moveDown' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "moveDown",
                callback: {
                    handler: this._moveDownCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowFunctionDAO
         * @method canMoveDown
         * Check if the function can move down in the callstack
         * This calls the method 'canMoveDown' of the server DAO 'WorkflowFunctionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowFunctionDAO",
                methodName: "canMoveDown",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_FUNCTION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
     },
     
    /**
     * Callback function called after {@link #addFunction} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _addFunctionCB: function(response)
    {
        var functionTarget = this._getFunctionTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: [functionTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #editFunction} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _editFunctionCB: function(response)
    {
        var functionTarget = this._getFunctionTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [functionTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #moveUp} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _moveUpCB: function(response)
    {
        if (response.message == "top-queue")
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_MOVE_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_MOVE_TOP_QUEUE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {
            var functionTarget = this._getFunctionTarget(response)
           
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: [functionTarget]
            });
            
            this._sendWorkflowModifiedMessage(response);
        }
    },
    
    /**
     * Callback function called after {@link #moveDown} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _moveDownCB: function(response)
    {
        if (response.message == "bottom-queue")
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_MOVE_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_MOVE_BOTTOM_QUEUE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {
            var functionTarget = this._getFunctionTarget(response)
           
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: [functionTarget]
            });
            
            this._sendWorkflowModifiedMessage(response);
        }
        
    },
    
    /**
     * Callback function called after {@link #deleteFunction} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _deleteFunctionCB: function(response)
    {
        var functionTarget = this._getFunctionTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.DELETED,
            targets: [functionTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * @private
     * Get function target
     * @param {Object} response the server response
     * @param {Number} response.workflowId the parent workflow id
     * @param {Number} response.stepId the parent step id
     * @param {Number} response.actionId the parent action id, can be null if direct parent is a step
     * @param {String} response.role the function's role 
     * @param {String} response.type the function's type
     * @param {Number} response.index the function's index in the prefunction or postfunction list
     * @returns {Object} the property's target 
     */
    _getFunctionTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_FUNCTION,
            parameters: {
                index: response.index,
                type: response.type,
                id: response.id,
                workflowId: response.workflowId,
                stepId: response.stepId,
                actionId: response.actionId,
                isLast: response.isLast
            }
        };
    },
    
    /**
     * @private
     * Get the action target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the parent step
     * @param {String} response.actionId the id of the action
     * @param {String} response.actionLabel the label of the action
     */
    _getActionTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_ACTION,
            parameters: {
                id: response.actionId,
                label:response.actionLabel,
                stepId: response.stepId,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Get the step target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the step
     * @param {String} response.stepLabel the label of the step
     */
    _getStepTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_STEP,
            parameters: {
                id: response.stepId,
                label:response.stepLabel,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Send a message modified for workflow target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     */
    _sendWorkflowModifiedMessage: function(response)
    {
        var workflowTarget = {
            id: Ametys.message.MessageTarget.WORKFLOW_OBJECT,
            parameters: {
                id: response.workflowId
            }
        };
        var stepTarget = this._getStepTarget(response);
        if (response.actionId != null)
        {
            stepTarget.subtargets = this._getActionTarget(response);
        }
        workflowTarget.subtargets = stepTarget;
            
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [workflowTarget]
        });
    }
});
