/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The workflow step DAO
 */
Ext.define('Ametys.plugins.workflow.dao.WorkflowStepDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowStepDAO
         * @method createStep 
         * Create workflow step
         * This calls the method 'createStep' of the server DAO 'WorkflowStepDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowStepDAO",
                methodName: "createStep",
                callback: {
                    handler: this._createStepCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_STEP_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowStepDAO
         * @method editStep 
         * Edit a workflow step
         * This calls the method 'editStep' of the server DAO 'WorkflowStepDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowStepDAO",
                methodName: "editStep",
                callback: {
                    handler: this._editStepCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_STEP_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowStepDAO
         * @method editStepLabel
         * Edit a workflow step label
         * This calls the method 'editStepLabel' of the server DAO 'WorkflowStepDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowStepDAO",
                methodName: "editStepLabel",
                callback: {
                    handler: this._editStepCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_STEP_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowStepDAO
         * @method deleteStep 
         * Delete a workflow step
         * This calls the method 'deleteStep' of the server DAO 'WorkflowStepDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowStepDAO",
                methodName: "deleteStep",
                callback: {
                    handler: this._deleteStepCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_STEP_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowStepDAO
         * @method getStepInfos 
         * Get the step editable infos 
         * This calls the method 'getStepInfos' of the server DAO 'WorkflowStepDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowStepDAO",
                methodName: "getStepInfos",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_STEP_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowStepDAO
         * @method hasSteps 
         * Is true if workflow has steps
         * This calls the method 'hasSteps' of the server DAO 'WorkflowStepDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowStepDAO",
                methodName: "hasSteps",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_STEP_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * Callback function called after {@link #createStep} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _createStepCB: function(response)
    {
        if (response.message == 'duplicate-id')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_CREATE_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_CREATE_STEP_DIALOG_ID_ERROR}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {   
             var workflowTarget = this._getWorkflowTarget(response);
             workflowTarget.subtargets = this._getStepTarget(response);
            
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.CREATED,
                targets: [workflowTarget]
            });
        }
    },
    
    /**
     * Callback function called after {@link #editStep} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _editStepCB: function(response)
    {
        if (response.message == 'incoming-actions')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_EDIT_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_INCOMING_ACTIONS_EDIT_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else if (response.message == 'duplicate-id')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_EDIT_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_STEP_DUPLICATE_ID_EDIT_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {   
             var workflowTarget = this._getWorkflowTarget(response);
             workflowTarget.subtargets = this._getStepTarget(response);
            
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.MODIFIED,
                targets: [workflowTarget]
            });
        }
    },
        
    /**
     * Callback function called after {@link #deleteStep} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _deleteStepCB: function(response)
    {
        if (response.message == 'incoming-actions')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_DELETE_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_INCOMING_ACTIONS_DELETE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {   
            var workflowTarget = this._getWorkflowTarget(response);
            
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.MODIFIED,
                targets: [workflowTarget]
            });
        }
    },
    
    /**
     * @private
     * Get the workflow target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     */
    _getWorkflowTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_OBJECT,
            parameters: {
                id: response.workflowId
            }
        }
    },
    
    /**
     * @private
     * Get the step target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the step
     * @param {String} response.stepLabels the label of the step
     */
    _getStepTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_STEP,
            parameters: {
                id: response.stepId,
                label: response.stepLabels,
                workflowId: response.workflowId
            }
        };
    }
});
