/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.ametys.plugins.workflow.component.WorkflowArgument;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.FunctionProvider;

/**
 * Interface for adding description and arguments to {@link FunctionProvider}
 */
public interface EnhancedFunction extends FunctionProvider
{
    /** The order of execution for the function */
    public static enum FunctionType
    {
        /** Function being executed before step change */
        PRE,
        /** Function being executed after step change */
        POST,
        /** Function that can be executed both before or after step change */
        BOTH
    }
    
    /**
     * Get the list of accepted arguments for this function
     * @return a List of argument names and associated description
     */
    public default List<WorkflowArgument> getArguments()
    {
        return new ArrayList<>();
    }
    
    /**
     * Get the label for this function
     * @return the label
     */
    public I18nizableText getLabel();
    
    /**
     * Get the function label depending on arguments values
     * @param argumentsValues a map of the arguments with their values in current workflow
     * @return a label to display in workflow editor vue
     */
    public default I18nizableText getFullLabel(Map<String, String> argumentsValues)
    {
        return getLabel();
    }
    
    /**
     * Return the type for the function
     * @return the type 's code
     */
    public default FunctionType getFunctionExecType()
    {
        return  FunctionType.BOTH;
    }
    
    /**
     * Get the function's visibilities depending on rights
     * @return a list of all the allowed right profiles
     */
    public default List<WorkflowVisibility> getVisibilities()
    {
        ArrayList<WorkflowVisibility> visibilityList = new ArrayList<>();
        visibilityList.add(WorkflowVisibility.SYSTEM); //By default, none of the enhanced function can be visible without system rights 
        return visibilityList;
    }
}
