/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.dao;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.plugins.workflow.EnhancedFunction.FunctionType;
import org.ametys.plugins.workflow.EnhancedFunctionExtensionPoint;
import org.ametys.plugins.workflow.ModelItemTypeExtensionPoint;
import org.ametys.plugins.workflow.component.WorkflowArgument;
import org.ametys.plugins.workflow.support.AvalonTypeResolver;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowElementDefinitionHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.SimpleViewItemGroup;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.disableconditions.DisableCondition;
import org.ametys.runtime.model.disableconditions.DisableCondition.OPERATOR;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.disableconditions.DisableConditions.ASSOCIATION_TYPE;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.FunctionProvider;
import com.opensymphony.workflow.TypeResolver;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.DescriptorFactory;
import com.opensymphony.workflow.loader.FunctionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * DAO for workflow element's pre and pos functions
 */
public class WorkflowFunctionDAO extends AbstractLogEnabled implements Component, Serviceable
{
    /** Extension point for workflow arguments data type */
    protected static ModelItemTypeExtensionPoint _workflowArgumentDataTypeExtensionPoint;
    
    private static final String __FUNCTION_DEFAULT_TYPE = "avalon";
    private static final String __ATTRIBUTE_FUNCTIONS_LIST = "functions-list";
    private static final String __ATTRIBUTE_FUNCTION_TYPES = "function-types";
    
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The workflow step DAO */
    protected WorkflowStepDAO _workflowStepDAO;
    
    /** The workflow transition DAO */
    protected WorkflowTransitionDAO _workflowTransitionDAO;
    
    /** The service manager */
    protected ServiceManager _manager;

    /** Extension point for EnhancedFunctions */
    protected EnhancedFunctionExtensionPoint _enhancedFunctionExtensionPoint;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _workflowSessionHelper = (WorkflowSessionHelper) smanager.lookup(WorkflowSessionHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) smanager.lookup(WorflowRightHelper.ROLE);
        _workflowStepDAO = (WorkflowStepDAO) smanager.lookup(WorkflowStepDAO.ROLE);
        _workflowTransitionDAO = (WorkflowTransitionDAO) smanager.lookup(WorkflowTransitionDAO.ROLE);
        _enhancedFunctionExtensionPoint = (EnhancedFunctionExtensionPoint) smanager.lookup(EnhancedFunctionExtensionPoint.ROLE);
        _workflowArgumentDataTypeExtensionPoint = (ModelItemTypeExtensionPoint) smanager.lookup(ModelItemTypeExtensionPoint.ROLE_WORKFLOW);
        _manager = smanager;
    }
    
    /**
     * Get the function's parameters as fields to configure edition form panel
     * @return the parameters field as Json readable map
     * @throws ProcessingException exception while saxing view to json
     */
    @Callable(rights = {"Workflow_Right_Edit", "Workflow_Right_Edit_User"})
    public Map<String, Object> getFunctionsModel() throws ProcessingException 
    {
        Map<String, Object> response = new HashMap<>();
        
        View view = new View();
        SimpleViewItemGroup fieldset = new SimpleViewItemGroup();
        fieldset.setName("functions");
        
        Set<Pair<String, EnhancedFunction>> enhancedFunctions = _enhancedFunctionExtensionPoint.getAllFunctions()
                .stream()
                .filter(this::_hasFunctionRight)
                .collect(Collectors.toSet());
        ElementDefinition<String> functionsList = _getFunctionListModelItem(enhancedFunctions);
        List<ElementDefinition> argumentModelItems = _getArgumentsAndTypeModelItems(enhancedFunctions);
        
        ViewElement functionListView = new ViewElement();
        functionListView.setDefinition(functionsList);
        fieldset.addViewItem(functionListView);
        
        for (ElementDefinition functionArgument : argumentModelItems)
        {
            ViewElement argumentView = new ViewElement();
            argumentView.setDefinition(functionArgument);
            fieldset.addViewItem(argumentView);
        }
        
        view.addViewItem(fieldset);
        
        response.put("parameters", view.toJSON(DefinitionContext.newInstance().withEdition(true)));
        
        return response;
    }
    
    private boolean _hasFunctionRight(Pair<String, EnhancedFunction> function)
    {
        List<WorkflowVisibility> functionVisibilities = function.getRight().getVisibilities();
        if (functionVisibilities.contains(WorkflowVisibility.USER))
        {
            return _workflowRightHelper.hasEditUserRight();
        }
        else if (functionVisibilities.contains(WorkflowVisibility.SYSTEM))
        {
            return _workflowRightHelper.hasEditSystemRight();
        }
        return false;
    }

    /**
     * Get a list of workflow arguments model items with disable conditions on non related function selected
     * @param enhancedFunctions a list of Pair with id and enhanced function
     * @return the list of model items
     */
    protected List<ElementDefinition> _getArgumentsAndTypeModelItems(Set<Pair<String, EnhancedFunction>> enhancedFunctions)
    {
        List<ElementDefinition> argumentModelItems = new ArrayList<>();
        for (Pair<String, EnhancedFunction> functionPair : enhancedFunctions)
        {
            String functionId = functionPair.getLeft();
            
            DisableCondition disableCondition = new DisableCondition(__ATTRIBUTE_FUNCTIONS_LIST, OPERATOR.NEQ, functionId); 
            
            EnhancedFunction function = functionPair.getRight();
            if (function.getFunctionExecType().equals(FunctionType.BOTH))
            {
                argumentModelItems.add(_getFunctionTypeModelItem(functionId, disableCondition));
            }
            
            for (WorkflowArgument arg : function.getArguments())
            {
                arg.setName(functionId + "-" + arg.getName());
                DisableConditions disableConditions = arg.getDisableConditions();
                if (disableConditions == null)
                {
                    disableConditions = new DisableConditions();
                }
                disableConditions.setAssociation(ASSOCIATION_TYPE.OR);
                disableConditions.getConditions().add(disableCondition);
                arg.setDisableConditions(disableConditions);
                argumentModelItems.add(arg);
            }
        }
        return argumentModelItems;
    }
    
    /**
     * Get the model item for the list of functions
     * @param enhancedFunctions the list of enhanced functions
     * @return an enum of the functions as a model item 
     */
    protected ElementDefinition<String> _getFunctionListModelItem(Set<Pair<String, EnhancedFunction>> enhancedFunctions)
    {
        ElementDefinition<String> functionsList = WorkflowElementDefinitionHelper.getElementDefinition(
            __ATTRIBUTE_FUNCTIONS_LIST,
            new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_FUNCTION_DIALOG_ROLE"), 
            new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_FUNCTION_DIALOG_ROLE_DESC"),
            true, 
            false
        );
        StaticEnumerator<String> functionsStaticEnumerator = new StaticEnumerator<>();
        for (Pair<String, EnhancedFunction> function : enhancedFunctions)
        {
            functionsStaticEnumerator.add(function.getRight().getLabel(), function.getLeft());
        }
        functionsList.setEnumerator(functionsStaticEnumerator);
        
        return functionsList;
    }

    /**
     * Get the view item for the fonctions types
     * @param functionId id of current function
     * @param disableCondition the condition for disabling the field
     * @return the view element
     */
    protected ElementDefinition<String> _getFunctionTypeModelItem(String functionId, DisableCondition disableCondition)
    {
        DisableConditions disableConditions = new DisableConditions();
        disableConditions.getConditions().add(disableCondition);
        
        StaticEnumerator<String> functionTypeStaticEnum = new StaticEnumerator<>();
        functionTypeStaticEnum.add(new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_FUNCTION_DIALOG_PREFUNCTION_TYPE"), FunctionType.PRE.name());
        functionTypeStaticEnum.add(new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_FUNCTION_DIALOG_POSTFUNCTION_TYPE"), FunctionType.POST.name());
        
        ElementDefinition<String> functionTypes = WorkflowElementDefinitionHelper.getElementDefinition(
                __ATTRIBUTE_FUNCTION_TYPES + "-" + functionId,
                new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_FUNCTION_DIALOG_TYPES_LABEL"),
                new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_ADD_FUNCTION_DIALOG_TYPES_DESC"),
                true,
                false
            );
        functionTypes.setDisableConditions(disableConditions);
        functionTypes.setEnumerator(functionTypeStaticEnum);
        
        return functionTypes;
    }
    
    /**
     * Get the function's parameters as a json view and their current values
     * @param workflowName the workflow's unique name
     * @param stepId the parent step's id
     * @param actionId the parent action's id can be null
     * @param type whether the function is a pre of post function
     * @param id the function's id
     * @param index the function's index in the pre/post function list
     * @return map of the function's infos
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getFunctionParametersValues(String workflowName, Integer stepId, Integer actionId, String type, String id, Integer index) 
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        Map<String, String> args = new HashMap<>();
        
        FunctionDescriptor function = _getFunction(workflowDescriptor, stepId, actionId, type, index);
        
        //change arguments key to make them unique
        Map<String, String> arguments = function.getArgs();
        for (Entry<String, String> entry : arguments.entrySet())
        {
            String argName = entry.getKey().equals("id") ? __ATTRIBUTE_FUNCTIONS_LIST : id + "-" + entry.getKey();
            args.put(argName, entry.getValue());
        }
        
        Map<String, Object> results = new HashMap<>();
        results.put("parametersValues", args);
        return results;
    }

    private FunctionDescriptor _getFunction(WorkflowDescriptor workflowDescriptor, Integer stepId, Integer actionId, String type, Integer index)
    {
        List<FunctionDescriptor> functions = _getTypedFunctions(workflowDescriptor, stepId, actionId, type);
        FunctionDescriptor function = functions.get(index);
        return function;
    }

    /**
     * Add a function to the workflow element
     * @param workflowName the workflow's unique name
     * @param stepId the parent step's id
     * @param actionId the parent action's id can be null
     * @param params Map of the function arguments
     * @return map of the function's infos
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> addFunction(String workflowName, Integer stepId, Integer actionId, Map<String, Object> params)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        String functionId = (String) params.get(__ATTRIBUTE_FUNCTIONS_LIST);
        EnhancedFunction enhancedFunction = _enhancedFunctionExtensionPoint.getExtension(functionId);
        
        String functionType = enhancedFunction.getFunctionExecType().equals(FunctionType.BOTH) 
                ? (String) params.get(__ATTRIBUTE_FUNCTION_TYPES + "-" + functionId) 
                : enhancedFunction.getFunctionExecType().name();
        
        List<FunctionDescriptor> functions = _getTypedFunctions(workflowDescriptor, stepId, actionId, functionType);
        Map<String, String> functionParams = _getFunctionParamsValuesAsString(enhancedFunction, functionId, params);
        FunctionDescriptor newFunction = _createFunctionDescriptor(functionParams);
        functions.add(newFunction);
        
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getFunctionInfos(workflowDescriptor, stepId, actionId, functionType, functionId, functions.size() - 1, true);
    }

    private FunctionDescriptor _createFunctionDescriptor(Map<String, String> functionParams)
    {
        DescriptorFactory factory = new DescriptorFactory();
        FunctionDescriptor newFunction = factory.createFunctionDescriptor();
        newFunction.setType(__FUNCTION_DEFAULT_TYPE);
        newFunction.getArgs().putAll(functionParams);
        
        return newFunction;
    }

    /**
     * Get the list of arguments as String, parse multiple arguments
     * @param enhancedFunction the current function class
     * @param functionId the function's id
     * @param params List of function arguments with values
     * @return the map of arguments formated for condition descriptor
     */
    protected Map<String, String> _getFunctionParamsValuesAsString(EnhancedFunction enhancedFunction, String functionId, Map<String, Object> params)
    {
        Map<String, String> functionParams = new HashMap<>();
        functionParams.put("id", functionId);
        List<WorkflowArgument> arguments = enhancedFunction.getArguments();
        if (!arguments.isEmpty())
        {
            for (WorkflowArgument argument : arguments)
            {
                String paramKey = functionId + "-" + argument.getName();
                String paramValue = argument.isMultiple()
                        ?  _getListAsString(params, paramKey)
                        : (String) params.get(paramKey);
                if (StringUtils.isNotBlank(paramValue))
                {
                    functionParams.put(argument.getName(), paramValue);
                }
            }
        }
        return functionParams;
    }

    @SuppressWarnings("unchecked")
    private String _getListAsString(Map<String, Object> params, String paramKey)
    {
        List<String> values = (List<String>) params.get(paramKey);
        return values == null ? "" : String.join(",", values);
    }

    /**
     * Get the list where belong a function can be either the prefunction list or the postfunctions
     * @param workflowDescriptor the current workflow
     * @param stepId id of current step
     * @param actionId id of current action can be null if selection is a step
     * @param functionType the type of function
     * @return the typed list
     */
    protected List<FunctionDescriptor> _getTypedFunctions(WorkflowDescriptor workflowDescriptor, Integer stepId, Integer actionId, String functionType)
    {
        List<FunctionDescriptor> functions = new ArrayList<>();
        if (actionId != null)
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            functions = functionType.equals(FunctionType.PRE.name()) ? action.getPreFunctions() : action.getPostFunctions();
        }
        else
        {
            StepDescriptor step = workflowDescriptor.getStep(stepId);
            functions = functionType.equals(FunctionType.PRE.name()) ? step.getPreFunctions() : step.getPostFunctions();
        }
        return functions;
    }
    
    /**
     * Edit the function
     * @param workflowName the workflow's unique name
     * @param stepId the parent step's id
     * @param actionId the parent action's id can be null
     * @param oldType the saved type for this fonction
     * @param params Map of the function arguments
     * @param indexOfFunction the function's index in the pre/post function list
     * @return map of the function's infos
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> editFunction(String workflowName, Integer stepId, Integer actionId, String oldType, Map<String, Object> params, int indexOfFunction)
    {
        WorkflowDescriptor workflow = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        _workflowRightHelper.checkEditRight(workflow);
    
        List<FunctionDescriptor> functions = _getTypedFunctions(workflow, stepId, actionId, oldType);
        FunctionDescriptor function = functions.get(indexOfFunction);
        
        int index = indexOfFunction;
        String functionId = (String) params.get(__ATTRIBUTE_FUNCTIONS_LIST);
        EnhancedFunction enhancedFunction = _enhancedFunctionExtensionPoint.getExtension(functionId);
        
        String functionType = (String) params.get(__ATTRIBUTE_FUNCTION_TYPES + "-" + functionId);
        String newType = StringUtils.isBlank(functionType) ? enhancedFunction.getFunctionExecType().name() : functionType;
        
        if (!oldType.equals(newType)) //Move function into the other typed list
        {
            functions.remove(function);
            functions = _getTypedFunctions(workflow, stepId, actionId, newType);
            functions.add(function);
            index = functions.size() - 1;
        }
        
        _updateFunctionArguments(function, functionId, enhancedFunction, params);
        _workflowSessionHelper.updateWorkflowDescriptor(workflow);
        
        return _getFunctionInfos(workflow, stepId, actionId, newType, functionId, index, _isLast(functions, indexOfFunction));
    }
    
    private void _updateFunctionArguments(FunctionDescriptor function, String functionId, EnhancedFunction enhancedFunction, Map<String, Object> params)
    {
        Map<String, String> args = function.getArgs();
        args.clear();
        args.putAll(_getFunctionParamsValuesAsString(enhancedFunction, functionId, params));
    }
    
    /**
     * Remove the function from its parent 
     * @param workflowName the workflow's unique name
     * @param stepId the parent step's id
     * @param actionId the parent action's id can be null
     * @param functionType whether the function is a pre of post function
     * @param id the function's id
     * @param indexToRemove the function's index in the pre/post function list
     * @return map of the function's infos
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> deleteFunction(String workflowName, Integer stepId, Integer actionId, String functionType, String id, int indexToRemove)
    {
        WorkflowDescriptor workflow = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        _workflowRightHelper.checkEditRight(workflow);
        List<FunctionDescriptor> functions = _getTypedFunctions(workflow, stepId, actionId, functionType);
        functions.remove(indexToRemove);
        _workflowSessionHelper.updateWorkflowDescriptor(workflow);
        
        return _getFunctionInfos(workflow, stepId, actionId, functionType, id, indexToRemove, _isLast(functions, indexToRemove));
    }
    
    /**
     * Swap the function with the one before it in the parent's pre/post function list
     * @param workflowName the workflow's unique name
     * @param stepId the parent step's id
     * @param actionId the parent action's id can be null
     * @param functionType whether the function is a pre of post function
     * @param functionIndex the function's index in the pre/post function list
     * @return map of the function's infos
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> moveUp(String workflowName, Integer stepId, Integer actionId, String functionType, int functionIndex)
    {
        WorkflowDescriptor workflow = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        _workflowRightHelper.checkEditRight(workflow);
        if (functionIndex == 0)
        {
            return Map.of("message", "top-queue");
        }
        List<FunctionDescriptor> functions = _getTypedFunctions(workflow, stepId, actionId, functionType);
        FunctionDescriptor functionToMove = functions.get(functionIndex);
        Collections.swap(functions, functionIndex, functionIndex - 1);
        _workflowSessionHelper.updateWorkflowDescriptor(workflow);
        
        return _getFunctionInfos(workflow, stepId, actionId, functionType, (String) functionToMove.getArgs().get("id"), functionIndex - 1, false);
    }
    
    /**
     * Swap the function with the one after it in the parent's pre/post function list
     * @param workflowName the workflow's unique name
     * @param stepId the parent step's id
     * @param actionId the parent action's id can be null
     * @param functionType whether the function is a pre of post function
     * @param functionIndex the function's index in the pre/post function list
     * @return map of the function's infos
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> moveDown(String workflowName, Integer stepId, Integer actionId, String functionType, int functionIndex)
    {
        WorkflowDescriptor workflow = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        _workflowRightHelper.checkEditRight(workflow);
        List<FunctionDescriptor> functions = _getTypedFunctions(workflow, stepId, actionId, functionType);
        if (functionIndex == functions.size() - 1)
        {
            return Map.of("message", "bottom-queue");
        }
        FunctionDescriptor functionToMove = functions.get(functionIndex);
        Collections.swap(functions, functionIndex, functionIndex + 1);
        _workflowSessionHelper.updateWorkflowDescriptor(workflow);
        
        return _getFunctionInfos(workflow, stepId, actionId, functionType, (String) functionToMove.getArgs().get("id"), functionIndex + 1, _isLast(functions, functionIndex + 1));
    }
    
    /**
     * Check if function can have bigger index in its parent's pre/post function list
     * @param functions the current list of functions
     * @param functionIndex the function's index in the pre/post function list
     * @return true if the function can move down
     */
    protected boolean _isLast(List<FunctionDescriptor> functions, int functionIndex)
    {
        return functionIndex == functions.size() - 1;
    }
    
    /**
     * Get pre and postfunctions of current step
     * @param workflowName the workflow unique name
     * @param stepId id of the current step
     * @return a list of the step's functions
     */
    @SuppressWarnings("unchecked")
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getStepFunctions(String workflowName, Integer stepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        List<Map<String, Object>> functions2json = new ArrayList<>();
        if (_workflowRightHelper.canRead(workflowDescriptor))
        {
            StepDescriptor step = workflowDescriptor.getStep(stepId);
            if (step != null)
            {
                functions2json.addAll(_functions2JSON(step.getPreFunctions(), FunctionType.PRE.name()));
                functions2json.addAll(_functions2JSON(step.getPostFunctions(), FunctionType.POST.name()));
            }
        }
        return Map.of("data", functions2json);
    }
    
    /**
     * Get pre and postfunctions of current action
     * @param workflowName the workflow unique name
     * @param actionId id of the current action
     * @return a list of the action's functions
     */
    @SuppressWarnings("unchecked")
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> getActionFunctions(String workflowName, Integer actionId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        List<Map<String, Object>> functions2json = new ArrayList<>();
        if (_workflowRightHelper.canRead(workflowDescriptor))
        {
            ActionDescriptor action = workflowDescriptor.getAction(actionId);
            if (action != null)
            {
                functions2json.addAll(_functions2JSON(action.getPreFunctions(), FunctionType.PRE.name()));
                functions2json.addAll(_functions2JSON(action.getPostFunctions(), FunctionType.POST.name()));
            }
        }
        
        return Map.of("data", functions2json);
    }
    
    private List<Map<String, Object>> _functions2JSON(List<FunctionDescriptor> functions, String type)
    {
        List<Map<String, Object>> functions2json = new ArrayList<>();
        for (int index = 0; index < functions.size(); index++)
        {
            functions2json.add(_function2JSON(functions.get(index), type, index, _isLast(functions, index)));
        }
        return functions2json;
    }
    
    private Map<String, Object> _function2JSON(FunctionDescriptor workflowFunction, String type, int index, boolean isLast)
    {
        String id = (String) workflowFunction.getArgs().get("id");
        Map<String, Object> functionInfos = new HashMap<>();
        functionInfos.put("type", type);
        functionInfos.put("id", id);
        functionInfos.put("index", index);
        functionInfos.put("isLast", isLast);
        
        try
        {
            TypeResolver typeResolver = new AvalonTypeResolver(_manager);
            FunctionProvider function = typeResolver.getFunction(workflowFunction.getType(), workflowFunction.getArgs());
            if (function instanceof EnhancedFunction enhancedFunction)
            {
                I18nizableText description = _getFunctionDescription(workflowFunction, enhancedFunction);
                if (description != null)
                {
                    functionInfos.put("description", description);
                }
            }
        }
        catch (WorkflowException e)
        {
            getLogger().error("Function " + id + " couldn't be resolved", e);
        }
        return functionInfos;
    }

    private I18nizableText _getFunctionDescription(FunctionDescriptor workflowFunction, EnhancedFunction enhancedFunction)
    {
        List<WorkflowArgument> arguments = enhancedFunction.getArguments();
        Map<String, String> values = new HashMap<>();
        for (WorkflowArgument arg : arguments)
        {
            values.put(arg.getName(), (String) workflowFunction.getArgs().get(arg.getName()));
        }
        
        I18nizableText description = enhancedFunction.getFullLabel(values);
        return description;
    }
    
    /**
     * Get the function's description
     * @param workflowName the workflow's unique name
     * @param stepId the parent step's id
     * @param actionId the parent action's id can be null
     * @param type whether the function is a pre of post function
     * @param id the enhanced function's id
     * @param index the function's index in the pre/post function list
     * @return the function's description
     */
    @Callable(rights = Callable.SKIP_BUILTIN_CHECK)
    public I18nizableText getFunctionDescription(String workflowName, Integer stepId, Integer actionId, String type, String id, Integer index)
    {
        WorkflowDescriptor workflow = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        _workflowRightHelper.checkReadRight(workflow);
        FunctionDescriptor function = _getFunction(workflow, stepId, actionId, type, index);
        EnhancedFunction enhancedFunction = _enhancedFunctionExtensionPoint.getExtension(id);
        return _getFunctionDescription(function, enhancedFunction);
    }
    
    private Map<String, Object> _getFunctionInfos(WorkflowDescriptor workflow, Integer stepId, Integer actionId, String type, String id, int index, boolean isLast)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("workflowId", workflow.getName());
        results.put("stepId", stepId);
        results.put("stepLabel", _workflowStepDAO.getStepLabel(workflow, stepId));
        if (actionId != null)
        {
            ActionDescriptor action = workflow.getAction(actionId);
            results.put("actionId", actionId);
            results.put("actionLabel", _workflowTransitionDAO.getActionLabel(action));
        }
        results.put("type", type);
        results.put("id", id);
        results.put("index", index);
        results.put("isLast", isLast);
        return results;
    }

}
