/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.definition;

import java.io.File;

import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.plugin.Feature;
import org.ametys.runtime.plugin.Plugin;
import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * The {@link WorkflowDefinition} extension point to list all available workflows.
 */
public class WorkflowDefinitionExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<WorkflowDefinition>
{
    /** The Avalon role */
    public static final String ROLE = WorkflowDefinitionExtensionPoint.class.getName();
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
    }
    
    @Override
    public void initializeExtensions() throws Exception
    {
        /* Add custom workflow found in WEB-INF/param/workflows and ignore those which are already defined in the kernel */
        
        // List workflow files
        File[] workflowFiles = _workflowHelper.getParamWorkflowDir().listFiles((file, name) -> name.endsWith(".xml"));

        if (workflowFiles != null)
        {
            for (File workflowFile : workflowFiles)
            {
                String workflowId = _getWorkflowIdFromFilename(workflowFile.getName());
                
                // If the workflow definition is not already declared, add it to known extensions
                if (!hasExtension(workflowId))
                {
                    _addNewExtension(workflowId);
                }
            }
        }
        
    }
    
    private void _addNewExtension(String workflowId) throws ConfigurationException
    {
        DefaultConfiguration extensionConf = new DefaultConfiguration("extension");
        extensionConf.setAttribute("class", AutomaticWorkflowDefinition.class.getName());
        
        // Set null to the configuration because the configuration is get into the method
        addExtension(workflowId, Plugin.DEFAULT_NAME, Feature.DEFAULT_NAME, extensionConf);
    }
    
    /**
     * Add a workflow to the workflow definition extension point
     * @param workflowId id of workflow
     * @throws ConfigurationException exception while trying to configure workflow 
     */
    public void addOrUpdateExtension(String workflowId) throws ConfigurationException
    {
        if (!hasExtension(workflowId))
        {
            // If the workflow definition is not already declared, add it to known extensions
            _addNewExtension(workflowId);
        }
        else
        {
            // Otherwise, force the workflow reloading
            getExtension(workflowId).reset();
        }
    }
    
    /**
     * Get the workflow ID from the workflow filename.
     * @param filename The workflow filename
     * @return the workflow ID
     */
    protected String _getWorkflowIdFromFilename(String filename)
    {
        return filename.substring(0, filename.lastIndexOf('.'));
    }
}
