/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Properties;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.ModifiableSource;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceNotFoundException;
import org.apache.excalibur.source.SourceResolver;
import org.apache.xml.serializer.OutputPropertiesFactory;
import org.xml.sax.SAXException;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.workflow.component.WorkflowLanguageManager;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for saxing i18n catalogs
 */
public class I18nHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The helper role */
    public static final String ROLE = I18nHelper.class.getName();
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
   
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow language manager */
    protected WorkflowLanguageManager _workflowLanguageManager;
    
    /** The source resolver */
    protected SourceResolver _sourceResolver;
    
    /** The observation manager */
    protected ObservationManager _observationManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _workflowSessionHelper = (WorkflowSessionHelper) manager.lookup(WorkflowSessionHelper.ROLE);
        _workflowLanguageManager = (WorkflowLanguageManager) manager.lookup(WorkflowLanguageManager.ROLE);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Get the default language to use for the i18n translation
     * @return a language code
     */
    public String getI18nDefaultLanguage()
    {
        Optional<String> fileLanguage = Optional.empty();
        try
        {
            // Parse default catalog application to know the default language
            fileLanguage = _getFileLanguage("application");
        }
        catch (Exception e)
        {
            getLogger().warn("An exception occured while getting current catalog language", e);
        }
        return fileLanguage.orElseGet(_workflowLanguageManager::getCurrentLanguage);
    }
    
    /**
     * Get language used by XML file
     * @param catalogName the catalog name
     * @return the language code
     * @throws Exception while resolving and getting file configuration
     */
    private Optional<String> _getFileLanguage(String catalogName) throws Exception
    {
        String defaultLanguage = null;
        String defaultI18nCatalogPath = getPrefixCatalogLocation(catalogName) + ".xml";
        Source defaultI18nCatalogSource = null;
        try
        {
            defaultI18nCatalogSource = _sourceResolver.resolveURI(defaultI18nCatalogPath);
            if (defaultI18nCatalogSource.exists())
            {
                try (InputStream is = defaultI18nCatalogSource.getInputStream())
                {
                    defaultLanguage = new DefaultConfigurationBuilder(true)
                            .build(is)
                            .getAttribute("xml:lang", null);
                }
            }
            
        }
        catch (SourceNotFoundException e)
        {
            getLogger().warn("Couldn't find file at path: {} a new file will be created", defaultI18nCatalogPath, e);
        }
        finally
        {
            _sourceResolver.release(defaultI18nCatalogSource);
        }
        
        return Optional.ofNullable(defaultLanguage).filter(StringUtils::isNotEmpty);
    }
    
    /**
     * Translate i18n label for workflow element, return a default name if translation is not found
     * @param workflowName the workflow's unique name
     * @param i18nKey an i18n key pointing to workflow element's label 
     * @param defaultKey a default i18n key for workflow element
     * @return a translated label
     */
    public String translateKey(String workflowName, I18nizableText i18nKey, I18nizableText defaultKey)
    {
        return Optional.of(workflowName)
            .map(_workflowSessionHelper::getTranslations)
            .map(translations -> translations.get(i18nKey))
            // if current language doesn't have translations jump to _translateKey()
            .map(translation -> translation.get(_workflowLanguageManager.getCurrentLanguage()))
            .filter(StringUtils::isNotEmpty)
            .orElseGet(() -> _translateKey(i18nKey, defaultKey));
    }
    
    private String _translateKey(I18nizableText i18nKey, I18nizableText defaultKey)
    {
        return Optional.ofNullable(_i18nUtils.translate(i18nKey))
                .filter(StringUtils::isNotBlank)
                .orElseGet(() -> _i18nUtils.translate(defaultKey));
    }
    
    /**
     * Transform workflow name in unique I18n key
     * @param workflowName the workflow's unique name
     * @return the workflow label I18n Key
     */
    public I18nizableText getWorkflowLabelKey(String workflowName)
    {
        return ArrayUtils.contains(_workflowHelper.getWorkflowNames(), workflowName)
                ? _workflowHelper.getWorkflowLabel(workflowName) 
                : new I18nizableText("application", buildI18nWorkflowKey(workflowName));
    }
    
    /**
     * Generate unique key for workflow element label
     * @param workflowName the workflow's unique name
     * @param type the workflow's element type
     * @param workflowElementId the element's id
     * @return a unique i18n key 
     */
    public I18nizableText generateI18nKey(String workflowName, String type, int workflowElementId)
    {
        String key = buildI18nWorkflowKey(workflowName) + "_" + type.toUpperCase() + "_" + workflowElementId;
        String workflowCatalog = _workflowHelper.getWorkflowCatalog(workflowName);
        I18nizableText i18nKey = new I18nizableText(workflowCatalog, key);
        return i18nKey;
    }

    /**
     * Sax new messages into i18N catalogs
     * @param newI18nMessages catalog of the new i18N messages, key is language, value is map of i18nKeys, translation 
     * @param currentCatalog the current catalog
     * @throws Exception exception while reading file
     */
    public void saveCatalogs(Map<String, Map<I18nizableText, String>> newI18nMessages, String currentCatalog) throws Exception
    {
        String prefixCatalogPath = getPrefixCatalogLocation(currentCatalog);
        
        // Determine the default catalog language
        String defaultLanguage = getI18nDefaultLanguage();
        Map<String, Map<I18nizableText, String>> i18nCatalogs = new HashMap<>();
        for (Entry<String, Map<I18nizableText, String>> i18nMessageTranslation : newI18nMessages.entrySet())
        {
            Map<I18nizableText, String> i18nMessages = new HashMap<>();
            String language = i18nMessageTranslation.getKey();
            String catalogPath = prefixCatalogPath + (currentCatalog.equals("application") && language.equals(defaultLanguage) ? "" : ("_" + language)) + ".xml";
            i18nMessages = readI18nCatalog(i18nCatalogs, catalogPath, currentCatalog);
            _updateI18nMessages(i18nMessages, i18nMessageTranslation.getValue());
            _saveI18nCatalog(i18nMessages, catalogPath, language);
        }
    }
    
    /**
     * Clear the i18n caches (ametys and cocoon)
     */
    public void clearCaches()
    {
        _i18nUtils.reloadCatalogues();
        
        _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_CACHE_RESET,
                _currentUserProvider.getUser(),
                Collections.singletonMap(org.ametys.core.ObservationConstants.ARGS_CACHE_ID, I18nUtils.I18N_CACHE)));
    }
    
    /**
     * Generate i18n catalog from workflow's new i18n translations
     * @param translationsToConvert the workflow to save's translation list: keys are future i18n key, values are maps of language codes and translations 
     * @return a map of i18n catalogs : keys are languages, values are pair of i18nkey, translation
     */
    public Map<String, Map<I18nizableText, String>> createNewI18nCatalogs(Map<I18nizableText, Map<String, String>> translationsToConvert)
    {
        Map<String, Map<I18nizableText, String>> i18nMessageTranslations = new HashMap<>();
        for (Entry<I18nizableText, Map<String, String>> newI18n : translationsToConvert.entrySet())
        {
            _updateI18nMessageTranslations(i18nMessageTranslations, newI18n.getKey(), newI18n.getValue());
        }
        return i18nMessageTranslations;
    }

    /**
     * Add translations to catalogs
     * @param i18nMessageTranslations the map of i18n catalogs, keys are languages, values are pair of i18nkey, translation
     * @param key an i18nKey to add
     * @param translations a map of translations to transform: key is language and value is translation
     */
    private void _updateI18nMessageTranslations(Map<String, Map<I18nizableText, String>> i18nMessageTranslations, I18nizableText key, Map<String, String> translations)
    {
        for (Entry<String, String> translation : translations.entrySet())
        {
            String language = translation.getKey();
            Map<I18nizableText, String> map = i18nMessageTranslations.computeIfAbsent(language, __ -> new HashMap<>());
            map.put(key, translation.getValue());
        }
    }
    
    /**
     * Read existing i18n catalog
     * @param i18nCatalogs map of already read catalogs
     * @param path the path to the file to read
     * @param currentCatalog the catalog of the current workflow
     * @return the messages in the catalog as map
     */
    public Map<I18nizableText, String> readI18nCatalog(Map<String, Map<I18nizableText, String>> i18nCatalogs, String path, String currentCatalog)
    {
        if (i18nCatalogs.containsKey(path))
        {
            return i18nCatalogs.get(path);
        }
        
        Map<I18nizableText, String> i18nMessages = new LinkedHashMap<>();
        Source i18nCatalogSource = null;
        
        try
        {
            i18nCatalogSource = _sourceResolver.resolveURI(path);
            if (i18nCatalogSource.exists())
            {
                try (InputStream is = i18nCatalogSource.getInputStream())
                {
                    SAXParserFactory.newInstance().newSAXParser().parse(is, new I18nMessageHandler(i18nMessages, currentCatalog));
                    i18nCatalogs.put(path, i18nMessages);
                }
            }
        }
        catch (IOException | SAXException | ParserConfigurationException e)
        {
            getLogger().error("An error occured while reading existing i18n catalog", e);
        }
        finally
        {
            _sourceResolver.release(i18nCatalogSource);
        }
        
        return i18nMessages;
    }
    
    /**
     * Update current i18n catalog with new values
     * @param i18nMessages the current i18n catalog
     * @param newI18nMessages the map of new messages
     */
    private void _updateI18nMessages(Map<I18nizableText, String> i18nMessages, Map<I18nizableText, String> newI18nMessages)
    {
        for (Entry<I18nizableText, String> newI18nMessage : newI18nMessages.entrySet())
        {
            i18nMessages.put(newI18nMessage.getKey(), newI18nMessage.getValue());
        }
    }
    
    /**
     * Save the i18n catalog with new values 
     * @param i18nMessages the messages to save
     * @param catalogPath the path to the catalog
     * @param language the language of the translations
     */
    protected void _saveI18nCatalog(Map<I18nizableText, String> i18nMessages, String catalogPath, String language)
    {
        ModifiableSource defaultI18nCatalogSource = null;
        try
        {
            defaultI18nCatalogSource = (ModifiableSource) _sourceResolver.resolveURI(catalogPath);
            try (OutputStream os = defaultI18nCatalogSource.getOutputStream())
            {
                // create a transformer for saving sax into a file
                TransformerHandler th = ((SAXTransformerFactory) TransformerFactory.newInstance()).newTransformerHandler();

                StreamResult result = new StreamResult(os);
                th.setResult(result);

                // create the format of result
                Properties format = new Properties();
                format.put(OutputKeys.METHOD, "xml");
                format.put(OutputKeys.INDENT, "yes");
                format.put(OutputKeys.ENCODING, "UTF-8");
                format.put(OutputPropertiesFactory.S_KEY_INDENT_AMOUNT, "2");
                th.getTransformer().setOutputProperties(format);

                // sax the config into the transformer
                _saxI18nCatalog(th, i18nMessages, language);

            }
            catch (Exception e)
            {
                getLogger().error("An error occured while saving the catalog values.", e);
            }
        }
        catch (IOException e)
        {
            getLogger().error("An error occured while getting i18n catalog", e);
        }
        finally
        {
            _sourceResolver.release(defaultI18nCatalogSource);
        }

    }
    
    private void _saxI18nCatalog(TransformerHandler handler, Map<I18nizableText, String> i18nMessages, String language) throws SAXException
    {
        handler.startDocument();
        AttributesImpl attribute = new AttributesImpl();
        attribute.addCDATAAttribute("xml:lang", language);
        XMLUtils.startElement(handler, "catalogue", attribute);
        for (Entry<I18nizableText, String> i18Message : i18nMessages.entrySet())
        {
            attribute = new AttributesImpl();
            attribute.addCDATAAttribute("key", i18Message.getKey().getKey());
            XMLUtils.createElement(handler, "message", attribute, i18Message.getValue());
        }
        XMLUtils.endElement(handler, "catalogue");
        handler.endDocument();
    }
    
    /**
     * Get the overridable catalog location for a plugin, workspace or application.
     * @param catalogName The catalog name
     * @return the location
     */
    public String getPrefixCatalogLocation(String catalogName)
    {
        if (catalogName.equals("application"))
        {
            return _i18nUtils.getApplicationCatalogLocation() + "/" + I18nUtils.APPLICATION;
        }
        
        String[] catalogParts = catalogName.split("\\.", 2);
        if (catalogParts.length != 2)
        {
            throw new IllegalArgumentException("The catalog name should be composed of two parts (like plugin.cms): " + catalogName);
        }
        
        // Transform plugin to plugins and workspace to workspaces
        return _i18nUtils.getOverridableCatalogLocation(catalogParts[0], catalogParts[1]) + "/" + I18nUtils.MESSAGES;
    }
    
    /**
     * Build an i18n key for workflows like "WORKFLOW_[WORKFLOW_NAME]"
     * @param workflowName the workflow name
     * @return the built key
     */
    
    public static String buildI18nWorkflowKey(String workflowName)
    {
        return "WORKFLOW_" + workflowName.replace("-", "_").toUpperCase();
    }
}
