/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;

import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Helper for managing workflow rights
 */
public class WorflowRightHelper implements Component, Serviceable
{

    /** The helper role */
    public static final String ROLE = WorflowRightHelper.class.getName();
    
    private static final String __WORKFLOW_RIGHT_READ_SYSTEM = "Workflow_Right_Read";
    
    private static final String __WORKFLOW_RIGHT_READ_USER = "Workflow_Right_Read_User";
    
    private static final String __WORKFLOW_RIGHT_EDIT_SYSTEM = "Workflow_Right_Edit";
    
    private static final String __WORKFLOW_RIGHT_EDIT_USER = "Workflow_Right_Edit_User";

    /** Rights manager available to subclasses. */
    protected RightManager _rightManager;

    /** The context */
    protected Context _context;
    
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _userProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Check if current user has read rights to access function for specific workflow, throws IllegalAccessError if not
     * @param workflow the current workflow
     */
    public void checkReadRight(WorkflowDescriptor workflow)
    {
        UserIdentity currentUser = _userProvider.getUser();
        if (!canRead(workflow))
        {
            throw new IllegalAccessError("User '" + currentUser + "' tried to handle workflows without convenient right ['" + __WORKFLOW_RIGHT_READ_USER + "' or '" + __WORKFLOW_RIGHT_READ_SYSTEM + "' or '" + __WORKFLOW_RIGHT_EDIT_USER + "' or '" + __WORKFLOW_RIGHT_EDIT_SYSTEM + "']");
        }
    }
    
    /**
     * Check if current user has edition rights to access function, throws IllegalAccessError if not
     */
    public void checkEditRight()
    {
        UserIdentity currentUser = _userProvider.getUser();
        if (!hasEditUserRight())
        {
            throw new IllegalAccessError("User '" + currentUser + "' tried to handle workflows without convenient right ['" + __WORKFLOW_RIGHT_EDIT_SYSTEM + "']");
        }
    }
    
    /**
     * Check if current user has edition rights to access function on current workflow, throws IllegalAccessError if not
     * @param workflow the current workflow
     */
    public void checkEditRight(WorkflowDescriptor workflow)
    {
        UserIdentity currentUser = _userProvider.getUser();
        if (!canWrite(workflow))
        {
            throw new IllegalAccessError("User '" + currentUser + "' tried to handle workflow " + workflow.getName() + " without convenient right ['" + __WORKFLOW_RIGHT_EDIT_SYSTEM + "']");
        }
    }
    
    /**
     * Check if user can edit current workflow
     * @param workflow the current workflow
     * @return true if user has the right needed to edit this workflow
     */
    public boolean canWrite(WorkflowDescriptor workflow)
    {
        Map<String, Object> metaAttributes = workflow.getMetaAttributes();
        return metaAttributes.containsKey("user") 
                ? hasEditUserRight()
                : hasEditSystemRight();
    }
    
    /**
     * Check if user can read current workflow
     * @param workflow the current workflow
     * @return true if user has the right needed to see this workflow
     */
    public boolean canRead(WorkflowDescriptor workflow)
    {
        Map<String, Object> metaAttributes = workflow.getMetaAttributes();
        return metaAttributes.containsKey("user") 
                ? _hasReadUserRight()
                : _hasReadSystemRight();
    }
    
    /**
     * Check if current user has edit right for workflow user
     * @return true if user has edition rights for workflows user or system
     */
    public boolean hasEditUserRight()
    {
        UserIdentity currentUser = _userProvider.getUser();
        return _rightManager.hasRight(currentUser, __WORKFLOW_RIGHT_EDIT_USER, "/${WorkspaceName}")  == RightResult.RIGHT_ALLOW || _rightManager.hasRight(currentUser, __WORKFLOW_RIGHT_EDIT_SYSTEM, "/${WorkspaceName}")  == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Check if current user has edit right for workflow system
     * @return true if user has edition rights for workflows system
     */
    public boolean hasEditSystemRight()
    {
        UserIdentity currentUser = _userProvider.getUser();
        return _rightManager.hasRight(currentUser, __WORKFLOW_RIGHT_EDIT_SYSTEM, "/${WorkspaceName}")  == RightResult.RIGHT_ALLOW;
    }
    
    private boolean _hasReadUserRight()
    {
        UserIdentity currentUser = _userProvider.getUser();
        return _rightManager.hasRight(currentUser, __WORKFLOW_RIGHT_READ_USER, "/${WorkspaceName}")  == RightResult.RIGHT_ALLOW || _rightManager.hasRight(currentUser, __WORKFLOW_RIGHT_READ_SYSTEM, "/${WorkspaceName}")  == RightResult.RIGHT_ALLOW  || hasEditUserRight();
    }
    
    private boolean _hasReadSystemRight()
    {
        UserIdentity currentUser = _userProvider.getUser();
        return _rightManager.hasRight(currentUser, __WORKFLOW_RIGHT_READ_SYSTEM, "/${WorkspaceName}")  == RightResult.RIGHT_ALLOW || hasEditSystemRight();
    }
}
