/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.plugins.workflow.AmetysWorkflowFactory;
import org.ametys.plugins.workflow.definition.WorkflowDefinition;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.AbstractWorkflow;
import com.opensymphony.workflow.FactoryException;
import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.loader.WorkflowFactory;
import com.opensymphony.workflow.spi.Step;
import com.opensymphony.workflow.spi.WorkflowStore;

/**
 * Helper to get information on the workflow structures
 */
public class WorkflowHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = WorkflowHelper.class.getName();
    
    /** The icon sizes */
    public static final String[] ICON_SIZES = new String[] {"small", "medium", "large"};
    
    /** The content types extension point */
    protected WorkflowProvider _workflowProvider;

    /** The source resolver */
    protected SourceResolver _sourceResolver;
    
    /** The context */
    protected Context _context;
    
    /** The Cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
        _sourceResolver = (SourceResolver) smanager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (org.apache.cocoon.environment.Context) _context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    /**
     * Enum of the types of workflow visibility profiles 
     */
    public static enum WorkflowVisibility
    {
        /** Element is visible for user with system rights */
        SYSTEM,
        /** Element is visible for user with user or system rights */
        USER
    }
    
    /**
     * Get a list of workflow names available
     * @return String[] an array of workflow names.
     */
    public String[] getWorkflowNames()
    {
        try
        {
            return _workflowProvider.getWorkflowFactory().getWorkflowNames();
        }
        catch (FactoryException e)
        {
            getLogger().error("Error getting workflow names", e);
        }
        
        return new String[0];
    }
    
    /**
     * Returns a workflow definition object associated with the given name.
     * @param workflowName the name of the workflow
     * @return the object graph that represents a workflow definition
     */
    public WorkflowDescriptor getWorkflowDescriptor(String workflowName)
    {
        try
        {
            return _workflowProvider.getWorkflowFactory().getWorkflow(workflowName);
        }
        catch (FactoryException e)
        {
            getLogger().error("Error loading workflow " + workflowName, e);
        }
        
        return null;
    }
    
    /**
     * Retrieves all actions of the workflow of a particular type of workflow
     * except initial actions.
     * @param workflowDesc the workflow descriptor.
     * @return all actions ids.
     * @throws IllegalArgumentException If the workflow name is not valid.
     */
    @SuppressWarnings("cast")
    public Set<Integer> getAllActions(WorkflowDescriptor workflowDesc)
    {
        Set<Integer> actions = new HashSet<>();
        
        // Add initial actions
        actions.addAll(_getActionIds((List<ActionDescriptor>) workflowDesc.getInitialActions()));
        
        // Add global actions
        actions.addAll(_getActionIds((List<ActionDescriptor>) workflowDesc.getGlobalActions()));
        
        // Add common actions
        actions.addAll(workflowDesc.getCommonActions().keySet());
        
        // Add steps actions
        List<Integer> stepActions = ((List<StepDescriptor>) workflowDesc.getSteps())
            .stream()
            .map(stepDesc -> (List<ActionDescriptor>) stepDesc.getActions())
            .map(this::_getActionIds)
            .flatMap(List::stream)
            .toList();
        actions.addAll(stepActions);
        
        return actions;
    }
    
    /**
     * Retrieves the name of an action.
     * @param workflowName The name of the workflow.
     * @param actionID The id of the action.
     * @return The name of the action or an empty string.
     */
    public String getActionName(String workflowName, int actionID)
    {
        return Optional.of(workflowName)
            .map(this::getWorkflowDescriptor)
            .map(workflowDesc -> workflowDesc.getAction(actionID))
            .map(ActionDescriptor::getName)
            .orElse(StringUtils.EMPTY);
    }
    
    /**
     * Retrieves the name of a step.
     * @param workflowName the name of the workflow.
     * @param stepId the id of the step.
     * @return the name of the step or an empty string.
     */
    public String getStepName(String workflowName, int stepId)
    {
        return Optional.of(workflowName)
                .map(this::getWorkflowDescriptor)
                .map(workflowDesc -> workflowDesc.getStep(stepId))
                .map(StepDescriptor::getName)
                .orElse(StringUtils.EMPTY);
    }
    
    /**
     * Retrieves the initial action id of a workflow.
     * @param workflowName the name of the workflow.
     * @return the first initial action id or <code>-1</code>
     *         if the workflow does not exist.
     */
    public int getInitialAction(String workflowName)
    {
        // Get the workflow descriptor
        WorkflowDescriptor workflowDesc = getWorkflowDescriptor(workflowName);
        if (workflowDesc == null)
        {
            return -1;
        }

        // Check if we have at least one initial action
        List actionDescriptor = workflowDesc.getInitialActions();
        if (actionDescriptor.isEmpty())
        {
            return -1;
        }

        // Return the first initial action
        return ((ActionDescriptor) actionDescriptor.get(0)).getId();
    }
    
    private List<Integer> _getActionIds(List<ActionDescriptor> actionDescriptors)
    {
        return actionDescriptors.stream()
                .map(ActionDescriptor::getId)
                .toList();
    }
    
    /**
     * Get the steps the workflow was "in" at a given date.
     * @param workflow workflow
     * @param entryId the workflow entry ID.
     * @param timestamp the date.
     * @return the list of steps the workflow was in.
     * @throws WorkflowException if an error occurs.
     */
    public List<Step> getStepAt(Workflow workflow, long entryId, Date timestamp) throws WorkflowException
    {
        return getStepsBetween(workflow, entryId, timestamp, timestamp);
    }
    
    /**
     * Get the steps the workflow was "in" between two dates.
     * @param workflow workflow
     * @param entryId the workflow entry ID.
     * @param start the start date.
     * @param end the end date.
     * @return the list of steps the workflow was in between the two dates.
     * @throws WorkflowException if an error occurs.
     */
    public List<Step> getStepsBetween(Workflow workflow, long entryId, Date start, Date end) throws WorkflowException
    {
        WorkflowStore store = ((AbstractWorkflow) workflow).getConfiguration().getWorkflowStore();
        
        List<Step> steps = new ArrayList<>();
        
        List<Step> allSteps = new ArrayList<>();
        
        allSteps.addAll(store.findCurrentSteps(entryId));
        allSteps.addAll(store.findHistorySteps(entryId));
        
        for (Step step : allSteps)
        {
            Date stepStartDate = step.getStartDate();
            Date stepFinishDate = step.getFinishDate();
            
            if (stepStartDate != null)
            {
                if (end.after(stepStartDate))
                {
                    if (stepFinishDate == null || start.before(stepFinishDate))
                    {
                        steps.add(step);
                    }
                }
            }
            else
            {
                if (stepFinishDate == null || start.before(stepFinishDate))
                {
                    steps.add(step);
                }
            }
        }
        
        return steps;
    }
    
    /**
     * Get the workflow label.
     * @param workflowName The name of the workflow
     * @return The label of the workflow with an i18n format.
     */
    public I18nizableText getWorkflowLabel(String workflowName)
    {
        return Optional.ofNullable(_getWorkflowDefinition(workflowName))
                       .map(WorkflowDefinition::getLabel)
                       .orElseGet(() -> new I18nizableText(workflowName));
    }
    
    /**
     * Get the catalog of the workflow.
     * @param workflowName The name of the workflow
     * @return The i18n catalog
     */
    public String getWorkflowCatalog(String workflowName)
    {
        WorkflowDefinition wfDef = _getWorkflowDefinition(workflowName);
        if (wfDef != null)
        {
            I18nizableText wfLabel = wfDef.getLabel();
            if (wfLabel.isI18n())
            {
                return wfLabel.getCatalogue();
            }
            return wfDef.getDefaultCatalog();
        }
        return "application";
    }
    
    private WorkflowDefinition _getWorkflowDefinition(String workflowName)
    {
        try
        {
            WorkflowFactory wFactory = _workflowProvider.getWorkflowFactory();
            if (wFactory instanceof AmetysWorkflowFactory ametysWFactory)
            {
                return ametysWFactory.getWorkflowDefinition(workflowName);
            }
        }
        catch (FactoryException e)
        {
            getLogger().debug("The workflow '{}' may not exist anymore", workflowName, e);
        }
        
        return null;
    }
    
    /**
     * Get the step descriptor from {@link WorkflowAwareAmetysObject}
     * @param ametysObject the targeted ametysObject
     * @param stepId the id of the step
     * @return the step descriptor or null if an error occurred
     */
    public StepDescriptor getStepDescriptor(WorkflowAwareAmetysObject ametysObject, int stepId)
    {
        long workflowId = ametysObject.getWorkflowId();
        
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(ametysObject);
        
        String workflowName = workflow.getWorkflowName(workflowId);
        WorkflowDescriptor workflowDescriptor = workflow.getWorkflowDescriptor(workflowName);
        
        if (workflowDescriptor != null)
        {
            StepDescriptor stepDescriptor = workflowDescriptor.getStep(stepId);
            if (stepDescriptor != null)
            {
                return stepDescriptor;
            }
            else if (getLogger().isWarnEnabled())
            {
                getLogger().warn("Unknown step id '" + stepId + "' for workflow for name : " + workflowName);
            }
        }
        else if (getLogger().isWarnEnabled())
        {
            getLogger().warn("Unknown workflow for name : " + workflowName);
        }
        
        return null;
    }
    
    /**
     * Serialize a step descriptor to JSON
     * @param step the step descriptor
     * @return a JSON map including the stepId, name and icons of the step
     */
    public Map<String, Object> workflowStep2JSON(StepDescriptor step)
    {
        Map<String, Object> workflowInfos = new LinkedHashMap<>();
        
        I18nizableText workflowStepName = new I18nizableText("application", step.getName());
        
        workflowInfos.put("stepId", step.getId());
        workflowInfos.put("name", workflowStepName);
        
        String workflowIconPath = _getWorkflowIconPath(workflowStepName.getCatalogue());
        for (String size : ICON_SIZES)
        {
            workflowInfos.put(size + "Icon", workflowIconPath + _getWorkflowIconFilename(workflowStepName, size));
        }
        
        return workflowInfos;
    }
    
    private String _getWorkflowIconPath(String catalog)
    {
        if ("application".equals(catalog))
        {
            return "/plugins/cms/resources_workflow/";
        }
    
        String[] catalogParts = catalog.split("\\.", 2);
        if (catalogParts.length != 2)
        {
            throw new IllegalArgumentException("The catalog name should be composed of two parts (like plugin.cms): " + catalog);
        }
        return "/" + catalogParts[0] + "s/" + catalogParts[1] + "/resources/img/workflow/";
    }

    private String _getWorkflowIconPathURI(String catalog)
    {
        if ("application".equals(catalog))
        {
            return "context://WEB-INF/param/workflow_resources/";
        }
        
        String[] catalogParts = catalog.split("\\.", 2);
        if (catalogParts.length != 2)
        {
            throw new IllegalArgumentException("The catalog name should be composed of two parts (like plugin.cms): " + catalog);
        }
        return catalogParts[0] + ":" + catalogParts[1] + "://resources/img/workflow/";
    }
    
    private String _getWorkflowIconFilename(I18nizableText label, String size)
    {
        return label.getKey() + "-" + size + ".png";
    }
    
    /**
     * Return a valid encoded path from i18n key of workflow element, or a default path if URI can't be resolved 
     * @param workflowElementName i18nKey of the workflowlabel, is used in the path of associated icon
     * @param defaultPath  path to default icon if uri can't be resolved
     * @return the  path to element icon 
     */
    public String getElementIconAsBase64(I18nizableText workflowElementName, String defaultPath) 
    {
        String workflowIconURI = getElementIconURI(workflowElementName);
        String path = _resolveImageAsBase64(workflowIconURI);
        if (path == null)
        {
            return _resolveImageAsBase64(defaultPath);
        }
        return path;
    }
    
    /**
     * Encode input stream in base64
     * @param path the path to the icon 
     * @return the encoded path 
     */
    protected String _resolveImageAsBase64(String path) 
    {
        Source source = null;
        try 
        {
            source = _sourceResolver.resolveURI(path);
            if (source.exists())
            {
                try (
                        InputStream dataIs = source.getInputStream();
                        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
                    )
                {
                    IOUtils.copy(dataIs, buffer);
                    String data = Base64.encodeBase64String(buffer.toByteArray());
                    return "data:" + source.getMimeType() + ";base64," + data;
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("couldn't resolve URI " + path, e);
        }
        finally
        {
            _sourceResolver.release(source);
        }
        
        return null;
    }
    
    /**
     * Return a valid path from i18n key of workflow element, or a default path if URI can't be resolved
     * @param workflowElementName i18nKey of the workflowlabel, is used in the path of associated icon
     * @param defaultPath path to default icon if uri can't be resolved
     * @return the resolved path to element icon
     */
    public String getElementIconPath(I18nizableText workflowElementName, String defaultPath) 
    {
        return ContextHelper.getRequest(_context).getContextPath() + _getElementIconPath(workflowElementName, defaultPath);
    }
    
    /**
     * Return a valid path from i18n key of workflow element, or default path if URI can't be resolved
     * @param workflowElementName i18nKey of the workflowlabel, is used in the path of associated icon
     * @param defaultPath path to default icon if uri can't be resolved
     * @return the path to element icon 
     */
    private String _getElementIconPath(I18nizableText workflowElementName, String defaultPath, String size) 
    {
        String workflowIconURI = getElementIconURI(workflowElementName);
        if (_imageURIExists(workflowIconURI))
        {
            return _getWorkflowIconPath(workflowElementName.getCatalogue()) + _getWorkflowIconFilename(workflowElementName, size);
        }
        return defaultPath;
    }
    
    private String _getElementIconPath(I18nizableText workflowElementName, String defaultPath) 
    {
        return _getElementIconPath(workflowElementName, defaultPath, "small");
    }
    
    /**
     * Get URI for an element icon
     * @param workflowElementName the name of the icon
     * @return the URI
     */
    public String getElementIconURI(I18nizableText workflowElementName) 
    {
        return _getWorkflowIconPathURI(workflowElementName.getCatalogue()) + _getWorkflowIconFilename(workflowElementName, "small");
    }
    
    /**
     * Get URI for an element icon
     * @param workflowElementName the name of the icon
     * @param size the wanted size for the icon
     * @return the URI
     */
    public String getElementIconURI(I18nizableText workflowElementName, String size) 
    {
        return _getWorkflowIconPathURI(workflowElementName.getCatalogue()) + _getWorkflowIconFilename(workflowElementName, size);
    }
    
    /**
     * Return if an image exist
     * @param path path to the image
     * @return true if image is found
     */
    protected boolean _imageURIExists(String path) 
    {
        Source resolveURI = null;
        try
        {
            resolveURI = _sourceResolver.resolveURI(path);
            return resolveURI.exists();
        }
        catch (Exception e) 
        {
            return false;
        }
        finally
        {
            _sourceResolver.release(resolveURI);
        }
    }
    
    /**
     * Get the param workflows directory.
     * @return the workflows directory in WEB-INF/param.
     */
    public File getParamWorkflowDir()
    {
        return new File(_cocoonContext.getRealPath("/WEB-INF/param/workflows"));
    }
}

