/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.readers;

import org.apache.commons.lang3.StringUtils;

/**
 * Abstract class for reading plantUML mindmaps 
 */
public abstract class AbstractPlantUMLMindmapSVGReader extends AbstractPlantUMLSVGReader
{
    @Override
    protected String _getPlantUMLType()
    {
        return "mindmap";
    }
    
    /**
     * Create a node in a plantUML mindmap
     * @param positionPrefix can be '+', '++' or '--' depending on the relative position of this node toward the central node
     * @param label the label of the node 
     * @param iconPath the encoded icon path of the step/action
     * @param color the background color for the node
     * @param jsFunction the function to insert in a link
     * @param linkTooltip the link tooltip
     * @param isBoxless true if the node doesn't have a border
     * @return the node graph as string
     */
    protected String _getMindMapNodeContent(String positionPrefix, String label, String iconPath, String color, String jsFunction, String linkTooltip, boolean isBoxless)
    {
        StringBuilder nodeContent = new StringBuilder(positionPrefix);
        
        if (isBoxless)
        {
            nodeContent.append("_");
        }
        
        if (StringUtils.isNotBlank(color))
        {
            nodeContent.append("[" + color + "]");
        }
        
        if (StringUtils.isNotBlank(jsFunction))
        {
            nodeContent.append(" [[javascript:parent." + jsFunction + " ");
            if (StringUtils.isNotBlank(linkTooltip)) 
            {
                nodeContent.append("{" + linkTooltip + "}");
            }
        }
        
        if (StringUtils.isNotBlank(iconPath))
        {
            nodeContent.append(" <img:" + iconPath + ">");
        }
        
        nodeContent.append(" " + label);
        
        if (StringUtils.isNotBlank(jsFunction))
        {
            nodeContent.append("]] ");
        }
        
        nodeContent.append(" \n");
        return nodeContent.toString();
    }
}
