/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.readers;

import java.util.Set;

import org.apache.cocoon.environment.Request;

import org.ametys.plugins.workflow.dao.WorkflowStepDAO;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Read action mindMap from plantUML
 */
public class PlantUmlMindMapSVGReader extends AbstractPlantUMLMindmapSVGReader 
{
    /** The color for incoming step nodes */
    public static final String __INCOMING_STEP_NODE_COLOR = "#fcf0f7";

    @Override
    protected String _getPlantUMLStyle()
    {
        StringBuilder style = new StringBuilder();
        
        style.append("<style> \n");
        style.append("mindmapDiagram { \n");
        style.append("node { \n");
        style.append("HyperLinkColor black \n");
        style.append("hyperlinkUnderlineThickness 0 \n"); //workaround until HyperlinkUnderline false is fixed
        style.append("} \n");
        style.append("} \n");
        style.append("</style> \n");
        
        return style.toString();
    }

    @Override
    protected String _getPlantUMLGraphContent(Request request, WorkflowDescriptor workflowDescriptor)
    {
        StringBuilder content = new StringBuilder();
        
        int actionId = Integer.valueOf((String) request.get("actionId"));
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        String workflowName = workflowDescriptor.getName();
        
        // Graph for central action node
        String actionContent = _getMindMapNodeContent(
                "+",
                _getActionLabel(workflowName, action),
                _workflowTransitionDAO.getActionIconPathAsBase64(workflowName, action),
                null,
                null,
                null,
                true
            );
        content.append(actionContent);
        
        // Graph for incoming steps
        Set<StepDescriptor> incomingSteps = _workflowStepDAO.getIncomingSteps(actionId, workflowDescriptor);
        for (StepDescriptor step : incomingSteps)
        {
            int stepId = step.getId();
            String stepLabel = _i18nHelper.translateKey(workflowName, _workflowStepDAO.getStepLabel(workflowDescriptor, stepId), WorkflowStepDAO.DEFAULT_STEP_NAME);
            stepLabel = _getStringWithEscapedSpace(stepLabel);
            String stepNode = _getMindMapNodeContent(
                "--",
                _getStepNodeLabel(workflowDescriptor, stepId),
                _workflowStepDAO.getStepIconPathAsBase64(workflowDescriptor, stepId),
                __INCOMING_STEP_NODE_COLOR,
                _getJsFunction(workflowName, String.valueOf(stepId), stepLabel, null, null),
                _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_LINK_SEE_STEP")),
                false
            );
            content.append(stepNode);
        }
        
        //Graph for outgoing steps
        Set<StepDescriptor> outgoingSteps = _workflowStepDAO.getOutgoingSteps(action, workflowDescriptor);
        for (StepDescriptor step : outgoingSteps)
        {
            int stepId = step.getId();
            String stepLabel = _i18nHelper.translateKey(workflowName, _workflowStepDAO.getStepLabel(workflowDescriptor, stepId), WorkflowStepDAO.DEFAULT_STEP_NAME);
            stepLabel = _getStringWithEscapedSpace(stepLabel);
            String stepNode = _getMindMapNodeContent(
                "++",
                _getStepNodeLabel(workflowDescriptor, stepId),
                _workflowStepDAO.getStepIconPathAsBase64(workflowDescriptor, stepId),
                __MAIN_STEP_NODE_COLOR,
                _getJsFunction(workflowName, String.valueOf(stepId), stepLabel, null, null),
                _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_LINK_SEE_STEP")),
                false
            );
            content.append(stepNode);
        }
        return content.toString();
    }
    
}
