/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.store;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;

import org.apache.jackrabbit.commons.JcrUtils;
import org.apache.jackrabbit.util.ISO8601;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.plugins.repository.AmetysRepositoryException;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.module.propertyset.memory.MemoryPropertySet;
import com.opensymphony.workflow.StoreException;
import com.opensymphony.workflow.query.Expression;
import com.opensymphony.workflow.query.FieldExpression;
import com.opensymphony.workflow.query.NestedExpression;
import com.opensymphony.workflow.query.WorkflowExpressionQuery;
import com.opensymphony.workflow.query.WorkflowQuery;
import com.opensymphony.workflow.spi.SimpleWorkflowEntry;
import com.opensymphony.workflow.spi.Step;
import com.opensymphony.workflow.spi.WorkflowEntry;

/**
 * Abstract workflow store for Jackrabbit
 */
@SuppressWarnings("deprecation")
public abstract class AbstractJackrabbitWorkflowStore implements AmetysWorkflowStore
{
    /** Namespace for node type names, node names and property names. */
    public static final String __NAMESPACE = "http://ametys.org/plugin/workflow/1.0";
    /** Prefix used for this namespace. */
    public static final String __NAMESPACE_PREFIX = "oswf";
    /** Prefix with colon for this namespace. */
    public static final String __NM_PREFIX = __NAMESPACE_PREFIX + ":";
    /** Root node type. */
    public static final String __ROOT_NT = __NM_PREFIX + "root";
    
    /** Entry node type. */
    static final String __ENTRY_NT = __NM_PREFIX + "entry";
    /** Step node type. */
    static final String __STEP_NT = __NM_PREFIX + "step";

    /** Root node name. */
    static final String __ROOT_NODE = __ROOT_NT;
    /** Entry node name prefix. */
    static final String __ENTRY_NODE_PREFIX = "workflow-";
    /** Current step node name. */
    static final String __CURRENT_STEP_NODE = __NM_PREFIX + "currentStep";
    /** History step node name. */
    static final String __HISTORY_STEP_NODE = __NM_PREFIX + "historyStep";

    /** Next entry id property for root node. */
    static final String __NEXT_ENTRY_ID_PROPERTY = __NM_PREFIX + "nextEntryId";
    /** ID for entry and step node. */
    static final String __ID_PROPERTY = __NM_PREFIX + "id";
    /** Workflow name property for entry node. */
    static final String __WF_NAME_PROPERTY = __NM_PREFIX + "workflowName";
    /** State property for entry node. */
    static final String __STATE_PROPERTY = __NM_PREFIX + "state";
    /** Next step id property for entry node. */
    static final String __NEXT_STEP_ID_PROPERTY = __NM_PREFIX + "nextStepId";
    
    /** Step id property for step node. */
    static final String __STEP_ID_PROPERTY = __NM_PREFIX + "stepId";
    /** Action id property for step node. */
    static final String __ACTION_ID_PROPERTY = __NM_PREFIX + "actionId";
    /** Owner property for step node. */
    static final String __OWNER_PROPERTY = __NM_PREFIX + "owner";
    /** Caller property for step node. */
    static final String __CALLER_PROPERTY = __NM_PREFIX + "caller";
    /** Start date property for step node. */
    static final String __START_DATE_PROPERTY = __NM_PREFIX + "startDate";
    /** Due date property for step node. */
    static final String __DUE_DATE_PROPERTY = __NM_PREFIX + "dueDate";
    /** Finish date property for step node. */
    static final String __FINISH_DATE_PROPERTY = __NM_PREFIX + "finishDate";
    /** Status property for step node. */
    static final String __STATUS_PROPERTY = __NM_PREFIX + "status";
    /** Previous steps ids property for step node. */
    static final String __PREVIOUS_STEPS_PROPERTY = __NM_PREFIX + "previousSteps";
    
    /** Log instance for logging events, errors, warnings, etc. */
    protected final Logger _log = LoggerFactory.getLogger(getClass());
    
    /** JCR Repository */
    protected Repository _repository;

    /**
     * Create a JackrabbitWorkflowStore.
     * @param repository the JCR Repository to use.
     */
    public AbstractJackrabbitWorkflowStore(Repository repository)
    {
        _repository = repository;
    }
    
    /**
     * Open a session to the _repository.
     * @return the session opened.
     * @throws RepositoryException if an error occurs.
     */
    protected Session _getSession() throws RepositoryException
    {
        return _repository.login();
    }
    
    /**
     * Release a session.<p>
     * Default implementation calls logout on the session.
     * @param session the session to release.
     */
    protected void _release(Session session)
    {
        if (session != null)
        {
            session.logout();
        }
    }
    
    /**
     * Convert a Date to a Calendar.
     * @param date The date to convert.
     * @return The date converted as a Calendar.
     */
    protected static Calendar __toCalendar(Date date)
    {
        if (date == null)
        {
            return null;
        }

        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        return calendar;
    }

    @Override
    public void init(Map props) throws StoreException
    {
        try
        {
            _createRootNode();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to initialize repository", e);
        }
    }
    
    /**
     * Create the root node.
     * @throws RepositoryException if an error occurs.
     */
    protected abstract void _createRootNode() throws RepositoryException;
    
    /**
     * Get the workflow store root node
     * @param session the session to use
     * @return The workflow store root node
     * @throws RepositoryException if an error occurs.
     */
    protected abstract Node _getRootNode(Session session) throws RepositoryException;
    
    @Override
    public void setEntryState(long entryId, int state) throws StoreException
    {
        Session session = null;

        try
        {
            session = _getSession();
            // Retrieve existing entry
            Node entry = getEntryNode(session, entryId);

            // Modify state
            entry.setProperty(__STATE_PROPERTY, state);

            session.save();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to change entry state for entryId: " + entryId, e);
        }
        finally
        {
            _release(session);
        }
    }

    @Override
    public PropertySet getPropertySet(long entryId) throws StoreException
    {
        // FIXME use a JcrPropertySet
        PropertySet ps = new MemoryPropertySet();
        ps.init(null, null);
        return ps;
        //throw new StoreException("PropertySet is not supported");
    }

    @Override
    public Step createCurrentStep(long entryId, int stepId, String owner, Date startDate, Date dueDate, String status, long[] previousIds) throws StoreException
    {
        Session session = null;
        try
        {
            session = _getSession();
            
            // Retrieve existing entry
            Node entry = getEntryNode(session, entryId);

            // Generate an unique step id inside the entry
            long id = _getNextStepId(entry);
            int actionId = 0;

            // Create a new current step node
            Node stepNode = entry.addNode(__CURRENT_STEP_NODE, __STEP_NT);
            stepNode.setProperty(__ID_PROPERTY, id);
            
            AmetysStep step = new AmetysStep(stepNode, this);
            
            step.setStepId(stepId);
            step.setActionId(actionId);
            step.setOwner(owner);
            step.setStartDate(startDate);
            
            if (dueDate != null)
            {
                step.setDueDate(dueDate);
            }
            
            step.setStatus(status);
            step.setPreviousStepIds(previousIds);
            
            step.save();
            
            return step;
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to store new entry", e);
        }
    }

    @Override
    public WorkflowEntry createEntry(String workflowName) throws StoreException
    {
        int state = WorkflowEntry.CREATED;
        
        // Generate an unique entry id
        long id;
        try
        {
            id = _getNextEntryId();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to store new entry", e);
        }
        
        WorkflowEntry workflowEntry = new SimpleWorkflowEntry(id, workflowName, state);
        
        // store in JCR
        storeNewEntry(workflowEntry);
        
        return workflowEntry;
        
    }
    
    /**
     * Store a new workflow entry into the JCR repository
     * @param workflowEntry The new entry
     * @throws StoreException on error
     */
    protected void storeNewEntry(WorkflowEntry workflowEntry) throws StoreException
    {
        Session session = null;
        
        String workflowName = workflowEntry.getWorkflowName();
        long id = workflowEntry.getId();
        int state = workflowEntry.getState();
        
        try
        {
            session = _getSession();
            
            // Retrieve workflow store root node and then the parent entry node
            Node root = _getRootNode(session);
            Node parentNode = _getOrCreateParentEntryNode(root, id);
            
            // Create entry node
            Node entryNode = parentNode.addNode(__ENTRY_NODE_PREFIX + id, __ENTRY_NT);
            
            if (_log.isDebugEnabled())
            {
                try
                {
                    _log.debug("Storing entry into path: " + entryNode.getPath());
                }
                catch (RepositoryException e)
                {
                    _log.warn("Unable to retrieve entry node path", e);
                }
            }
            
            entryNode.setProperty(__ID_PROPERTY, id);
            entryNode.setProperty(__WF_NAME_PROPERTY, workflowName);
            entryNode.setProperty(__STATE_PROPERTY, state);
            
            session.save();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to store new entry", e);
        }
        finally
        {
            _release(session);
        }
    }
    
    /**
     * Remove a workflow entry
     * @param entryId The id of workflow entry
     * @throws StoreException if an error occurred
     */
    public void removeEntry (long entryId) throws StoreException
    {
        Session session = null;
        
        try
        {
            session = _getSession();
            
            Node entryNode = getEntryNode(session, entryId);
            entryNode.remove();
            session.save();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to delete workflow entry " + entryId, e);
        }
        finally
        {
            _release(session);
        }
    }
    
    /**
     * Retrieves the parent node of a workflow entry.
     * Creates non existing ancestor nodes when necessary.
     * @param root The workflow store root node
     * @param id The workflow entry id
     * @return The parent node
     * @throws RepositoryException on repository error
     */
    protected abstract Node _getOrCreateParentEntryNode(Node root, long id) throws RepositoryException;

    /**
     * Retrieve an entry node from its id.
     * @param session the session to use.
     * @param entryId the id of the entry.
     * @return the entry node.
     * @throws RepositoryException if there is no entry for this id.
     */
    public Node getEntryNode(Session session, long entryId) throws RepositoryException
    {
        Node rootNode = _getRootNode(session);
        Node parentNode = _getOrCreateParentEntryNode(rootNode, entryId);

        return parentNode.getNode(__ENTRY_NODE_PREFIX + entryId);
    }

    /**
     * Retrieve an history step node from its id for a particular entry.
     * @param session The session to use.
     * @param entryId The id of the entry.
     * @param stepId The id of the step.
     * @return The step node.
     * @throws RepositoryException if no step matches or multiple steps match.
     */
    Node getHistoryStepNode(Session session, long entryId, long stepId) throws RepositoryException
    {
        Node entryNode = getEntryNode(session, entryId);
        NodeIterator itNode = entryNode.getNodes(__HISTORY_STEP_NODE);
        
        while (itNode.hasNext())
        {
            Node stepNode = itNode.nextNode();
            if (JcrUtils.getLongProperty(stepNode, __ID_PROPERTY, -1) == stepId)
            {
                return stepNode;
            }
        }
        
        throw new RepositoryException("Unknown entry node for entryId: " + entryId + " and stepId: " + stepId);
    }
    
    /**
     * Generate an unique entry id.
     * @return A new entry id.
     * @throws RepositoryException if an error occurs.
     */
    protected synchronized long _getNextEntryId() throws RepositoryException
    {
        Session session = null;

        try
        {
            // Use a new session to ensure that just this property is saved.
            session = _getSession();
            // Retrieve root node containing entries
            Node root = _getRootNode(session);
            long nextEntryId = root.getProperty(__NEXT_ENTRY_ID_PROPERTY).getLong();

            // Set the next entry id
            root.setProperty(__NEXT_ENTRY_ID_PROPERTY, nextEntryId + 1);
            session.save();

            // Return the previous entry id
            return nextEntryId;
        }
        finally
        {
            _release(session);
        }
    }


    /**
     * Generate an unique step id for an entry.
     * @param entry The entry.
     * @return A new step id.
     * @throws RepositoryException if an error occurs.
     */
    protected synchronized long _getNextStepId(Node entry) throws RepositoryException
    {
        long nextStepId = entry.getProperty(__NEXT_STEP_ID_PROPERTY).getLong();

        // Set the next step id
        entry.setProperty(__NEXT_STEP_ID_PROPERTY, nextStepId + 1);
        entry.getSession().save();

        // Return the previous step id
        return nextStepId;
    }

    @Override
    public List findCurrentSteps(long entryId) throws StoreException
    {
        List<Step> currentSteps = new ArrayList<>();
        Session session = null;
        
        try
        {
            session = _getSession();
            // Retrieve existing entry
            Node entry = getEntryNode(session, entryId);
            // Get current steps
            NodeIterator nodeIterator = entry.getNodes(__CURRENT_STEP_NODE);

            while (nodeIterator.hasNext())
            {
                // Convert each step node to a Step
                currentSteps.add(new AmetysStep(nodeIterator.nextNode(), this));
            }
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to change entry state for entryId: " + entryId, e);
        }
        
        return currentSteps;
    }
    
    @Override
    public WorkflowEntry findEntry(long entryId) throws StoreException
    {
        Session session = null;

        try
        {
            session = _getSession();
            // Retrieve existing entry
            Node entry = getEntryNode(session, entryId);

            String workflowName = entry.getProperty(__WF_NAME_PROPERTY).getString();
            int state = (int) entry.getProperty(__STATE_PROPERTY).getLong();

            return new SimpleWorkflowEntry(entryId, workflowName, state);
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to change entry state for entryId: " + entryId, e);
        }
        finally
        {
            _release(session);
        }
    }

    @Override
    public List findHistorySteps(long entryId) throws StoreException
    {
        List<Step> historySteps = new ArrayList<>();
        Session session = null;

        try
        {
            session = _getSession();
            // Retrieve existing entry
            Node entry = getEntryNode(session, entryId);
            // Get history steps
            NodeIterator nodeIterator = entry.getNodes(__HISTORY_STEP_NODE);

            while (nodeIterator.hasNext())
            {
                // Convert each step node to a Step
                historySteps.add(new AmetysStep(nodeIterator.nextNode(), this));
            }
            
            // Order by step descendant
            Collections.sort(historySteps, new Comparator<Step>()
            {
                @Override
                public int compare(Step step1, Step step2)
                {
                    return -Long.valueOf(step1.getId()).compareTo(step2.getId());
                }
            });
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to change entry state for entryId: " + entryId, e);
        }
        
        return historySteps;
    }

    @Override
    public Step markFinished(Step step, int actionId, Date finishDate, String status, String caller) throws StoreException
    {
        try
        {
            // Add new properties
            AmetysStep theStep = (AmetysStep) step;

            theStep.setActionId(actionId);
            theStep.setFinishDate(finishDate);
            theStep.setStatus(status);
            theStep.setCaller(caller);
            
            theStep.save();
            
            return theStep;
        }
        catch (AmetysRepositoryException e)
        {
            throw new StoreException("Unable to modify step for entryId: " + step.getEntryId() + " and stepId: " + step.getStepId(), e);
        }
    }

    @Override
    public void moveToHistory(Step step) throws StoreException
    {
        try
        {
            // Retrieve the step node
            Node stepNode = ((AmetysStep) step).getNode();
            Node entry = stepNode.getParent();

            // Get the existing path of the current node
            String currentStepPath = stepNode.getPath();
            // Set the destination path to history steps
            String historyStepPath = entry.getPath() + "/" + __HISTORY_STEP_NODE;
            // Move node
            stepNode.getSession().move(currentStepPath, historyStepPath);

            stepNode.getSession().save();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to move step to history for entryId: " + step.getEntryId() + " and stepId: " + step.getStepId(), e);
        }
    }

    @Override
    public List query(WorkflowQuery query) throws StoreException
    {
        List<Long> results = new ArrayList<>();
        Session session = null;

        try
        {
            session = _getSession();
            
            // Build XPath query
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            StringBuilder xPathQuery = new StringBuilder("//element(*, ");
            xPathQuery.append(__ENTRY_NT);
            xPathQuery.append(")");
            xPathQuery.append("[");
            xPathQuery.append(getCondition(query));
            xPathQuery.append("]");

            if (_log.isInfoEnabled())
            {
                _log.info("Executing xpath: " + xPathQuery);
            }

            Query jcrQuery = queryManager.createQuery(xPathQuery.toString(), Query.XPATH);
            QueryResult result = jcrQuery.execute();
            NodeIterator nodeIterator = result.getNodes();

            // Add matching entries
            while (nodeIterator.hasNext())
            {
                Node entry = nodeIterator.nextNode();
                results.add(entry.getProperty(__ID_PROPERTY).getLong());
            }
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to query entries", e);
        }
        finally
        {
            _release(session);
        }

        return results;
    }
    
    @Override
    public void clearHistory(long entryId) throws StoreException
    {
        Session session = null;

        try
        {
            session = _getSession();
            Node entryNode = getEntryNode(session, entryId);

            // Remove all history step nodes
            NodeIterator itNode = entryNode.getNodes(__HISTORY_STEP_NODE);
            
            while (itNode.hasNext())
            {
                Node historyStepNode = itNode.nextNode();
                String historyStepNodeIdentifier = historyStepNode.getIdentifier();
                // Remove references to this node
                PropertyIterator itProperty = historyStepNode.getReferences();
                
                while (itProperty.hasNext())
                {
                    Property property = itProperty.nextProperty();
                    Node parentNode = property.getParent();
                    String nodeTypeName = parentNode.getPrimaryNodeType().getName();
                    
                    if (nodeTypeName.equals(__STEP_NT))
                    {
                        // Remove current reference to this node
                        ValueFactory valueFactory = session.getValueFactory();
                        List<Value> newValues = new ArrayList<>();
                        Value[] values = property.getValues();
                        
                        for (Value value : values)
                        {
                            String identifier = value.getString();
                            
                            if (!identifier.equals(historyStepNodeIdentifier))
                            {
                                newValues.add(valueFactory.createValue(identifier));
                            }
                        }
                        
                        property.setValue(newValues.toArray(new Value[newValues.size()]));
                    }
                    else
                    {
                        // Not an expected reference
                        throw new RepositoryException("An history node is references by a unknown node of type: " + nodeTypeName);
                    }
                }
                
                // Remove history step node
                historyStepNode.remove();
            }
            
            session.save();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to clear entry history for entry id: " + entryId, e);
        }
        finally
        {
            _release(session);
        }
    }

    @Override
    public void deleteInstance(long entryId) throws StoreException
    {
        Session session = null;

        try
        {
            session = _getSession();
            Node entryNode = getEntryNode(session, entryId);
            
            // Remove entry node
            entryNode.remove();
            
            session.save();
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to delete instance for entry id: " + entryId, e);
        }
        finally
        {
            _release(session);
        }
    }

    /**
     * Get the condition from the query.
     * 
     * @param query The query.
     * @return The resulting condition.
     * @throws StoreException if the query is not valid.
     */
    protected String getCondition(WorkflowQuery query) throws StoreException
    {
        if (query.getLeft() == null)
        {
            // leaf node
            return buildFieldExpression(query);
        }
        else
        {
            int operator = query.getOperator();
            WorkflowQuery left = query.getLeft();
            WorkflowQuery right = query.getRight();

            switch (operator)
            {
                case WorkflowQuery.AND:
                    return '(' + getCondition(left) + ") and (" + getCondition(right) + ')';

                case WorkflowQuery.OR:
                    return '(' + getCondition(left) + ") or (" + getCondition(right) + ')';

                case WorkflowQuery.XOR:
                default:
                    throw new IllegalArgumentException("Not supported operator: " + operator);
            }
        }
    }

    /**
     * Build a field expression.
     * 
     * @param query The query.
     * @return The resulting field expression.
     * @throws StoreException if the query is not valid.
     */
    protected String buildFieldExpression(WorkflowQuery query) throws StoreException
    {
        StringBuilder condition = new StringBuilder();

        int qtype = query.getType();

        if (qtype == 0)
        { // then not set, so look in sub queries
            if (query.getLeft() != null)
            {
                qtype = query.getLeft().getType();
            }
        }

        if (qtype == WorkflowQuery.CURRENT)
        {
            condition.append(__CURRENT_STEP_NODE + "/");
        }
        else
        {
            condition.append(__HISTORY_STEP_NODE + "/");
        }

        Object value = query.getValue();
        int operator = query.getOperator();
        int field = query.getField();

        String oper;

        switch (operator)
        {
            case WorkflowQuery.EQUALS:
                oper = " = ";
                break;

            case WorkflowQuery.NOT_EQUALS:
                oper = " != ";
                break;

            case WorkflowQuery.GT:
                oper = " > ";
                break;

            case WorkflowQuery.LT:
                oper = " < ";
                break;

            default:
                oper = " = ";
        }

        String left = getPropertyName(field);
        String right = "";

        if (value == null)
        {
            if (operator == WorkflowQuery.EQUALS)
            {
                oper = "";
            }
            else if (operator == WorkflowQuery.NOT_EQUALS)
            {
                left = "not(" + left + ")";
                oper = "";
            }
            else
            {
                right = "null";
            }
        }
        else
        {
            right = translateValue(value);
        }

        condition.append(left);
        condition.append(oper);
        condition.append(right);

        return condition.toString();
    }

    @Override
    public List query(WorkflowExpressionQuery query) throws StoreException
    {
        List<Long> results = new ArrayList<>();
        Session session = null;

        try
        {
            session = _getSession();
            
            // Build XPath query
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            StringBuilder xPathQuery = new StringBuilder("//element(*, ");
            xPathQuery.append(__ENTRY_NT);
            xPathQuery.append(")");
            xPathQuery.append(getPredicate(query));
            xPathQuery.append(getSortCriteria(query));

            if (_log.isInfoEnabled())
            {
                _log.info("Executing xpath: " + xPathQuery);
            }

            Query jcrQuery = queryManager.createQuery(xPathQuery.toString(), Query.XPATH);
            QueryResult result = jcrQuery.execute();
            NodeIterator nodeIterator = result.getNodes();

            // Add matching entries
            while (nodeIterator.hasNext())
            {
                Node entry = nodeIterator.nextNode();
                results.add(entry.getProperty(__ID_PROPERTY).getLong());
            }
        }
        catch (RepositoryException e)
        {
            throw new StoreException("Unable to query entries", e);
        }
        finally
        {
            _release(session);
        }

        return results;
    }
    
    /**
     * Get the JCR sort criteria from the query.
     * 
     * @param query The query.
     * @return The sort criteria.
     * @throws StoreException if the criteria is not valid.
     */
    protected String getSortCriteria(WorkflowExpressionQuery query) throws StoreException
    {
        StringBuilder criteria = new StringBuilder();

        if (query.getSortOrder() != WorkflowExpressionQuery.SORT_NONE)
        {
            criteria.append(" order by ");

            // Set child axis for specific context (not entry context)
            // FIXME Use first field expression found for determining context
            FieldExpression fieldExpr = getFirstFieldExpression(query.getExpression());

            if (fieldExpr == null)
            {
                // No field expression, no sorting
                return "";
            }

            if (fieldExpr.getContext() == FieldExpression.CURRENT_STEPS)
            {
                criteria.append(__CURRENT_STEP_NODE + "/");
            }
            else if (fieldExpr.getContext() == FieldExpression.HISTORY_STEPS)
            {
                criteria.append(__HISTORY_STEP_NODE + "/");
            }

            criteria.append(getPropertyName(query.getOrderBy()));

            if (query.getSortOrder() == WorkflowExpressionQuery.SORT_DESC)
            {
                criteria.append(" descending");
            }
            else
            {
                criteria.append(" ascending");
            }
        }

        return criteria.toString();
    }
    
    /**
     * Retrieve the first field expression from an expression.
     * @param expression the expression to inspect.
     * @return the first field expression.
     */
    protected FieldExpression getFirstFieldExpression(Expression expression)
    {
        if (expression.isNested())
        {
            NestedExpression nestedExpr = (NestedExpression) expression;

            for (int i = 0; i < nestedExpr.getExpressionCount(); i++)
            {
                FieldExpression fieldExpr = getFirstFieldExpression(nestedExpr.getExpression(i));

                if (fieldExpr != null)
                {
                    // Field expression found
                    return fieldExpr;
                }
            }
        }
        else
        {
            // Field expression found
            return (FieldExpression) expression;
        }

        // No field expression found
        return null;
    }

    /**
     * Build a predicate from the query.
     * 
     * @param query The query.
     * @return The predicate.
     * @throws StoreException if the query is not valid.
     */
    protected String getPredicate(WorkflowExpressionQuery query) throws StoreException
    {
        Expression expression = query.getExpression();
        String predicate;

        if (expression.isNested())
        {
            predicate = buildNestedExpression((NestedExpression) expression);
        }
        else
        {
            predicate = buildFieldExpression((FieldExpression) expression);
        }

        return "[" + predicate + "]";
    }

    /**
     * Build a nested expression.
     * 
     * @param nestedExpr The nested expression.
     * @return The resulting condition.
     * @throws StoreException if the expression is not valid.
     */
    protected String buildNestedExpression(NestedExpression nestedExpr) throws StoreException
    {
        StringBuilder query = new StringBuilder();
        int exprCount = nestedExpr.getExpressionCount();

        for (int i = 0; i < exprCount; i++)
        {
            Expression subExpression = nestedExpr.getExpression(i);

            query.append("(");

            if (subExpression instanceof NestedExpression)
            {
                query.append(buildNestedExpression((NestedExpression) subExpression));
            }
            else if (subExpression instanceof FieldExpression)
            {
                query.append(buildFieldExpression((FieldExpression) subExpression));
            }

            query.append(")");

            if (i != exprCount - 1)
            {
                switch (nestedExpr.getExpressionOperator())
                {
                    case NestedExpression.OR:
                        query.append(" or ");
                        break;
                    case NestedExpression.AND:
                    default:
                        query.append(" and ");
                        break;
                }
            }
        }

        return query.toString();
    }

    /**
     * Build a field expression.
     * 
     * @param expr The field expression.
     * @return The resulting condition.
     * @throws StoreException if the expression is not valid.
     */
    private String buildFieldExpression(FieldExpression expr) throws StoreException
    {
        StringBuilder query = new StringBuilder();

        // Set child axis for specific context (not entry context)
        if (expr.getContext() == FieldExpression.CURRENT_STEPS)
        {
            query.append(__CURRENT_STEP_NODE + "/");
        }
        else if (expr.getContext() == FieldExpression.HISTORY_STEPS)
        {
            query.append(__HISTORY_STEP_NODE + "/");
        }

        Object value = expr.getValue();
        int operator = expr.getOperator();
        int field = expr.getField();

        String oper;

        switch (operator)
        {
            case FieldExpression.EQUALS:
                oper = " = ";
                break;

            case FieldExpression.NOT_EQUALS:
                oper = " != ";
                break;

            case FieldExpression.GT:
                oper = " > ";
                break;

            case FieldExpression.LT:
                oper = " < ";
                break;

            default:
                oper = " = ";
        }

        String left = getPropertyName(field);
        String right = "";

        if (value == null)
        {
            if (operator == FieldExpression.EQUALS)
            {
                oper = "";
            }
            else if (operator == FieldExpression.NOT_EQUALS)
            {
                left = "not(" + left + ")";
                oper = "";
            }
            else
            {
                right = "null";
            }
        }
        else
        {
            right = translateValue(value);
        }

        if (expr.isNegate())
        {
            query.append("not(");
        }

        query.append(left);
        query.append(oper);
        query.append(right);

        if (expr.isNegate())
        {
            query.append(")");
        }

        return query.toString();
    }

    /**
     * Convert and maybe escape a value to support XPath grammar.
     * 
     * @param value The value to translate.
     * @return The translated value.
     * @throws StoreException if the value cannot be translated.
     */
    protected String translateValue(Object value) throws StoreException
    {
        if (value instanceof Date)
        {
            Date date = (Date) value;
            Calendar calendar = new GregorianCalendar();
            calendar.setTime(date);
            return " xs:dateTime('" + ISO8601.format(calendar) + "')";
        }
        else if (value instanceof Integer || value instanceof Long)
        {
            // No need of escaping
            return value.toString();
        }

        // Escape single quote into two successive single quote
        return "'" + value.toString().replaceAll("'", "''") + "'";
    }

    /**
     * Retrieve a propertyName from a field id.
     * 
     * @param field The field id.
     * @return The corresponding property name.
     * @throws StoreException if the field is not valid.
     * @see com.opensymphony.workflow.query.FieldExpression
     */
    protected String getPropertyName(int field) throws StoreException
    {
        switch (field)
        {
            case FieldExpression.ACTION:
                return "@" + __ACTION_ID_PROPERTY;

            case FieldExpression.CALLER:
                return "@" + __CALLER_PROPERTY;

            case FieldExpression.FINISH_DATE:
                return "@" + __FINISH_DATE_PROPERTY;

            case FieldExpression.OWNER:
                return "@" + __OWNER_PROPERTY;

            case FieldExpression.START_DATE:
                return "@" + __START_DATE_PROPERTY;

            case FieldExpression.STEP:
                return "@" + __STEP_ID_PROPERTY;

            case FieldExpression.STATUS:
                return "@" + __STATUS_PROPERTY;

            case FieldExpression.STATE:
                return "@" + __STATE_PROPERTY;

            case FieldExpression.NAME:
                return "@" + __WF_NAME_PROPERTY;

            case FieldExpression.DUE_DATE:
                return "@" + __DUE_DATE_PROPERTY;

            default:
                throw new StoreException("Invalid field: " + field);
        }
    }
}
