/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.store;

import java.util.ArrayList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.cocoon.util.HashUtil;
import org.apache.commons.lang.StringUtils;

import org.ametys.plugins.workflow.support.WorkflowProvider;

/**
 * The generic workflow store.
 * @see WorkflowProvider
 */
public class GenericWorkflowStore extends AbstractJackrabbitWorkflowStore
{
    /** Internal hash tree node type. */
    static final String __HTREE_NT = __NM_PREFIX + "hashSegment";
    
    /**
     * Generic workflow store constructor
     * @param repository The repository
     */
    public GenericWorkflowStore(Repository repository)
    {
        super(repository);
    }

    @Override
    public boolean shouldClearHistory()
    {
        return true;
    }
    
    @Override
    protected void _createRootNode() throws RepositoryException
    {
        Session session = null;
        try
        {
            session = _getSession();
            
            Node root = session.getRootNode();
            if (!root.hasNode(__ROOT_NODE))
            {
                if (_log.isDebugEnabled())
                {
                    _log.debug("Creating osworkflow root node");
                }
                
                root.addNode(__ROOT_NODE, __ROOT_NT);
                
                session.save();
            }
            else
            {
                if (_log.isDebugEnabled())
                {
                    _log.debug("Existing osworkflow root node, skipping creation.");
                }
            }
        }
        finally
        {
            _release(session);
        }
    }
    
    @Override
    protected Node _getRootNode(Session session) throws RepositoryException
    {
        return session.getRootNode().getNode(__ROOT_NODE);
    }
    /**
     * Retrieve the hash used for an entry id.
     * @param entryId the entry id.
     * @return the hash parts.
     */
    protected List<String> _getEntryHash(long entryId)
    {
        long hash = Math.abs(HashUtil.hash(String.valueOf(entryId)));
        // Prepend 0 for having same length hash parts
        String hashHexa = StringUtils.leftPad(Long.toString(hash, 16), 4, "0");

        List<String> hashSegments = new ArrayList<>();
        
        hashSegments.add(hashHexa.substring(0, 2));
        hashSegments.add(hashHexa.substring(2, 4));
        
        return hashSegments;
    }
    
    @Override
    protected Node _getOrCreateParentEntryNode(Node root, long id) throws RepositoryException
    {
        // Balanced tree management
        Node htree = root;
        List<String> hashSegments = _getEntryHash(id);
        for (String hashSegment : hashSegments)
        {
            if (htree.hasNode(hashSegment))
            {
                // Go down to a leaf of the hash tree
                htree = htree.getNode(hashSegment);
            }
            else
            {
                // Add a new leaf to the hash tree
                htree = htree.addNode(hashSegment, __HTREE_NT);
            }
        }
        
        return htree;
    }
}
