/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.support;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Object for storing workflow into session, do not call outside of WorkflowSessionHelper
 */
class WorkflowToSave 
{
    private boolean _isCloned;
    private WorkflowDescriptor _workflow;
    private Map<I18nizableText, Map<String, String>> _translations;
    
    /** Map for storing new and old keys for cloned workflow element: key is the new key and value is the old one */
    private Map<I18nizableText, I18nizableText> _clonedKeys;
    
    /**
     * Init workflow to save
     * @param workflow a workflow descriptor
     * @param translations a map of translations for i18n label, key is i18n label, value is a map of language, translation
     */
    public WorkflowToSave(WorkflowDescriptor workflow, Map<I18nizableText, Map<String, String>> translations)
    {
        this._workflow = workflow;
        this._translations = translations;
    }
    
    /**
     * Get the workflow descriptor
     * @return a workflow descriptor
     */
    public WorkflowDescriptor getWorkflow()
    {
        return this._workflow;
    }
    
    /**
     * Set the workflow descriptor
     * @param workflowDescriptor a workflow descriptor
     */
    public void setWorkflow(WorkflowDescriptor workflowDescriptor)
    {
        this._workflow = workflowDescriptor;
    }
    
    /**
     * Get the the workflow to save's label translations
     * @return a map  with language as key and translation as value
     */
    public Map<I18nizableText, Map<String, String>> getTranslations()
    {
        return this._translations;
    }
    
    /**
     * Set the the workflow to save's label translations
     * @param translations the map of translations: key is i18n label, value is a map of language, translation
     */
    public void setTranslations(Map<I18nizableText, Map<String, String>> translations)
    {
        this._translations = translations;
    }
    
    /**
     * Update the workflow i18n catalogs
     * @param key the i18n key
     * @param labels the translations 
     */
    public void updateTranslations(I18nizableText key, Map<String, String> labels)
    {
        Map<String, String> cleanedLabels = _cleanEmptyEntries(labels);
        if (!cleanedLabels.isEmpty())
        {
            Map<String, String> translations = _translations.computeIfAbsent(key, __ -> new HashMap<>());
            translations.putAll(cleanedLabels);
        }
    }
    
    private Map<String, String> _cleanEmptyEntries(Map<String, String> entries)
    {
        return entries.entrySet()
                .stream()
                .filter(e -> StringUtils.isNotEmpty(e.getValue()))
                .collect(
                    Collectors.toMap(
                        Entry::getKey,
                        Entry::getValue
                    )
                );
    }
    
    /**
     * Get the isCloned property
     * @return true if the workflow is a clone and have not been saved
     */
    public boolean isCloned()
    {
        return this._isCloned;
    }
    
    /**
     * Set the isCloned property
     * @param isCloned the value to set
     */
    public void setCloned(boolean isCloned)
    {
        this._isCloned = isCloned;
    }
    
    /**
     * Get the map of cloned keys 
     * @return the cloned keys map
     */
    public Map<I18nizableText, I18nizableText> getClonedKeys()
    {
        return this._clonedKeys;
    }
    
    /**
     * Set the map of cloned keys
     * @param keys the map to set
     */
    public void setClonedKeys(Map<I18nizableText, I18nizableText> keys)
    {
        this._clonedKeys = keys;
    }
    
    
}
