/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow;

import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.plugins.workflow.store.AmetysStep;
import org.ametys.plugins.workflow.store.AmetysWorkflowStore;
import org.ametys.plugins.workflow.support.WorkflowHelper;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.AbstractWorkflow;
import com.opensymphony.workflow.StoreException;
import com.opensymphony.workflow.WorkflowContext;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.spi.Step;
import com.opensymphony.workflow.spi.WorkflowEntry;
import com.opensymphony.workflow.spi.WorkflowStore;

/**
 * Base Ametys class for manipulating workflow instances.
 */
public abstract class AbstractAmetysWorkflow extends AbstractWorkflow
{
    /** Logger available to subclasses. */
    protected static Logger _logger = LoggerFactory.getLogger(AbstractAmetysWorkflow.class);
    
    /** Reentrant locks for synchronizing access to each instance */
    protected Map<Long, ReentrantLock> _instancesLocks = new HashMap<>();
    
    /** workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /**
     * Constructor
     * @param workflowHelper The workflow helper
     * @param workflowContext The workflow context
     */
    protected AbstractAmetysWorkflow(WorkflowHelper workflowHelper, WorkflowContext workflowContext)
    {
        _workflowHelper = workflowHelper;
        context = workflowContext;
        
        _instancesLocks = new HashMap<>();
    }
    
    /**
     * Creates a new workflow instance of a particular type of workflow.
     * @param workflowName the type of the workflow.
     * @param initialActionId the initial action to perform or
     *        <code>Integer.MIN_VALUE</code> in order to use the first initial action.
     * @param inputs the arguments to transmit to the workflow. Can not be null.
     * @return the workflow instance id.
     */
    @Override
    public long initialize(String workflowName, int initialActionId, Map inputs) throws WorkflowException
    {
        int actionId = initialActionId;
        
        if (initialActionId == Integer.MIN_VALUE)
        {
            actionId = _workflowHelper.getInitialAction(workflowName);
        }
        
        Map<String, Object> initInputs = inputs;
        if (initInputs == null)
        {
            initInputs = new HashMap<>();
        }
        
        if (!initInputs.containsKey(AbstractWorkflowComponent.RESULT_MAP_KEY))
        {
            initInputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, new LinkedHashMap<>());
        }
        
        // TODO delete workflow entry if initialize has failed
        return super.initialize(workflowName, actionId, initInputs);
    }
    
    @Override
    public int[] getAvailableActions(long id, Map inputs)
    {
        Map<String, Object> doInputs = inputs;
        if (doInputs == null)
        {
            doInputs = new HashMap<>();
        }
        
        if (!doInputs.containsKey(AbstractWorkflowComponent.RESULT_MAP_KEY))
        {
            doInputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, new LinkedHashMap<>());
        }
        
        return super.getAvailableActions(id, inputs);
    }
    
    @Override
    @Deprecated
    @SuppressWarnings("unchecked")
    public int[] getAvailableActions(long id)
    {
        Map inputs = new HashMap();
        inputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, new LinkedHashMap<>());
        return super.getAvailableActions(id, inputs);
    }
    
    @Override
    public void doAction(long id, int actionId, Map inputs) throws WorkflowException
    {
        // Do not perform an action on the same workflow instance in the same time
        ReentrantLock instanceLock = null;
        
        // Récupérer le verrou sur l'instance
        synchronized (_instancesLocks)
        {
            instanceLock = _instancesLocks.get(id);
            
            // Check is a lock already exists
            if (instanceLock == null)
            {
                instanceLock = new ReentrantLock(true);
                // Store new lock
                _instancesLocks.put(id, instanceLock);
            }
        }
        
        try
        {
            Map<String, Object> doInputs = inputs;
            if (doInputs == null)
            {
                doInputs = new HashMap<>();
            }
            
            if (!doInputs.containsKey(AbstractWorkflowComponent.RESULT_MAP_KEY))
            {
                doInputs.put(AbstractWorkflowComponent.RESULT_MAP_KEY, new LinkedHashMap<>());
            }
            
            // Only one instance can perform an action on a given workflow instance
            instanceLock.lock();
            super.doAction(id, actionId, doInputs);
        }
        finally
        {
            // Allow another thread to perform an action on this workflow instance
            instanceLock.unlock();
        }
        
        // Vérifier si l'instance du workflow est terminée
        if (getEntryState(id) == WorkflowEntry.COMPLETED)
        {
            try
            {
                WorkflowStore store = getConfiguration().getWorkflowStore();
                
                if (store instanceof AmetysWorkflowStore)
                {
                    if (((AmetysWorkflowStore) store).shouldClearHistory())
                    {
                        // Supprimer l'instance de la base de données
                        // PERTE DE L'HISTORIQUE !
                        ((AmetysWorkflowStore) store).deleteInstance(id);
                    }
                }
            }
            catch (StoreException e)
            {
                _logger.error("Error while getting the workflow store", e);
            }
        }
    }
    
    @Override
    protected boolean transitionWorkflow(WorkflowEntry entry, List currentSteps, WorkflowStore store, WorkflowDescriptor wf, ActionDescriptor action, Map transientVars, Map inputs, PropertySet ps) throws WorkflowException
    {
        Date actionStartDate = new Date();
        
        // Mark all current steps before the action.
        for (Object step : currentSteps)
        {
            if (step instanceof AmetysStep)
            {
                AmetysStep theStep = (AmetysStep) step;
                theStep.setProperty("actionStartDate", actionStartDate);
                theStep.save();
            }
        }
        
        boolean complete = super.transitionWorkflow(entry, currentSteps, store, wf, action, transientVars, inputs, ps);
        
        // Mark the newly created step.
        Step createdStep = (Step) transientVars.get("createdStep");
        if (createdStep != null && createdStep instanceof AmetysStep)
        {
            AmetysStep theStep = (AmetysStep) createdStep;
            theStep.setProperty("actionFinishDate", new Date());
            theStep.save();
        }
        
        return complete;
    }
    
    @Override
    protected void checkImplicitFinish(ActionDescriptor action, long id) throws WorkflowException
    {
        // Do nothing ... we don't want to call completeEntry if the currentStep has no actions (so its a final step)
    }
}

