/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.cocoon;

import java.util.Collections;
import java.util.Map;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.environment.Redirector;

import com.opensymphony.workflow.InvalidInputException;
import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;

/**
 * Action for firing a transition on a workflow.<p>
 * The following parameters are supported:
 * <dl>
 *  <dt>workflowId
 *  <dd>the id of the workflow to use
 *  <dt>actionId
 *  <dd>the id of the action to fire
 * </dl>
 * {@link Map} returned is empty on successful transition
 * and throw an exception on error.
 */
public class WorkflowAction extends AbstractWorkflowAction
{
    @Override
    protected Map _act(Redirector redirector, Map objectModel, String source, Parameters parameters, int actionId, Map inputs) throws InvalidInputException, WorkflowException
    {
        Workflow workflow = _workflowProvider.getGenericWorkflow();
        workflow.doAction(_getWorkflowId(objectModel, source, parameters), actionId, inputs);
        return _getActionResult(redirector, objectModel, source, parameters);
    }

    /**
     * Provide the workflow id.<p>
     * Default implementation uses parameter <code>workflowId</code>.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @return the workflow id.
     */
    protected long _getWorkflowId(Map objectModel, String source, Parameters parameters)
    {
        try
        {
            return parameters.getParameterAsLong("workflowId");
        }
        catch (ParameterException e)
        {
            throw new RuntimeException("Missing parameter workflowId", e);
        }
    }
    
    /**
     * Provides the action result after successful workflow transition.<p>
     * Default implementation returns a empty {@link Map}.
     * @param redirector the redirector.
     * @param objectModel the current object model.
     * @param source the current source.
     * @param parameters the current parameters.
     * @return the action result.
     */
    protected Map _getActionResult(Redirector redirector, Map objectModel, String source, Parameters parameters)
    {
        return Collections.emptyMap();
    }
    
    @Override
    protected String _getExceptionContext(Map objectModel, String source, Parameters parameters)
    {
        return String.format("for the workflow of instance id: '%d'", _getWorkflowId(objectModel, source, parameters));
    }
}
