/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.component;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.Right;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.right.RightsExtensionPoint;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.EnhancedCondition;
import org.ametys.plugins.workflow.support.WorkflowElementDefinitionHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.StaticEnumerator;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Condition for checking rights of an user for the current action.<p>
 * The following configuration can be used for checking rights:<br>
 * <pre>
 * &lt;condition type="avalon"&gt;
 * &nbsp;&nbsp;&lt;arg name="role"&gt;org.ametys.plugins.workflow.component.CheckRightsCondition&lt;/arg&gt;
 * &nbsp;&nbsp;&lt;arg name="right"&gt;Right_Edition&lt;/arg&gt;
 * [&nbsp;&nbsp;&lt;arg name="context"&gt;/cms&lt;/arg&gt;]
 * &lt;/condition&gt;
 * </pre>
 */
public class CheckRightsCondition extends AbstractWorkflowComponent implements EnhancedCondition, Initializable, Disposable
{
    /**
     * Boolean to force the CheckRightsCondition to returns ok.
     */
    public static final String FORCE = CheckRightsCondition.class.getName() + "$force";
    
    /** Default context to use. */
    protected static final String __DEFAULT_CONTEXT = "/cms";
    /** Key for getting the right to check. */
    protected static final String __RIGHT_KEY = "right";
    /** Key for getting the context to use. */
    protected static final String __CONTEXT_KEY = "context";
    /** Rights manager available to subclasses. */
    protected RightManager _rightManager;
    /** The rights extension point */
    protected RightsExtensionPoint _rightsExtensionPoint;
    /** I18nUtils */
    protected I18nUtils _i18nUtils;
    
    @Override
    public void initialize() throws Exception
    {
        _rightManager = (RightManager) _manager.lookup(RightManager.ROLE);
        _rightsExtensionPoint = (RightsExtensionPoint) _manager.lookup(RightsExtensionPoint.ROLE);
        _i18nUtils = (I18nUtils) _manager.lookup(I18nUtils.ROLE);
    }

    @Override
    public boolean passesCondition(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        // Retrieve the right to check
        String rightNeeded = (String) args.get(__RIGHT_KEY);
        if (rightNeeded == null)
        {
            if (_logger.isWarnEnabled())
            {
                _logger.warn(String.format("Missing 'right' argument for workflow action id: %d, failing condition",
                                           transientVars.get("actionId")));
            }
            
            return false;
        }
        
        Boolean force = (Boolean) transientVars.get(FORCE);
        if (force != null && force)
        {
            // Pass condition
            return true;
        }
        
        // Retrieve the user
        UserIdentity user = getUser(transientVars);
        boolean passesCondition = _checkRights(transientVars, args, user, rightNeeded);
        
        if (!passesCondition)
        {
            List<ConditionFailure> conditionFailures = getConditionFailures(transientVars);
            if (conditionFailures != null)
            {
                conditionFailures.add(new ConditionFailure(
                        String.format("Check right condition failed for workflow action id %d, user '%s' and right '%s'", transientVars.get("actionId"), user, rightNeeded),
                        CheckRightsCondition.class.getName()
                ));
            }
            
            addWorkflowError(transientVars, new I18nizableText("plugin.workflow", "WORKFLOW_CHECK_RIGHTS_CONDITION_FAILED"));
        }
        return passesCondition;
    }

    /**
     * Check if the current user has the needed right.
     * @param transientVars variables that will not be persisted.
     * @param args the properties for this function invocation.
     * @param user the current user.
     * @param rightNeeded the needed right.
     * @return <code>true</code> if the user has the right, <code>false</code>
     *         otherwise.
     * @throws WorkflowException if an error occurs.
     */
    protected boolean _checkRights(Map transientVars, Map args, UserIdentity user, String rightNeeded) throws WorkflowException
    {
        // Compute the context
        Object context = _computeContext(transientVars, args, user, rightNeeded);

        // Check the context presence
        if (context == null)
        {
            if (_logger.isWarnEnabled())
            {
                _logger.warn(String.format("Missing context for checking rights for workflow action id: %d, failing condition",
                                           transientVars.get("actionId")));
            }
            
            return false;
        }
        
        if (rightNeeded.contains("|"))
        {
            return checkMultipleOrRights(user, StringUtils.split(rightNeeded, '|'), context);
        }
        else if (rightNeeded.contains("&"))
        {
            return checkMultipleAndRights(user, StringUtils.split(rightNeeded, '&'), context);
        }
        
        // Check if current user has the right
        return hasRight(user, rightNeeded, context);
    }
    
    /**
     * Check that a user has all the given rights on a context (AND condition).
     * @param user the user.
     * @param rights the rights to check.
     * @param context the right context.
     * @return true if the user has all the rights, false otherwise.
     */
    protected boolean checkMultipleAndRights(UserIdentity user, String[] rights, Object context)
    {
        for (String right : rights)
        {
            if (!hasRight(user, right, context))
            {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Check that a user has at least one of the given rights on a context (OR condition).
     * @param user the user.
     * @param rights the rights to check.
     * @param context the right context.
     * @return true if the user has at least one right, false otherwise.
     */
    protected boolean checkMultipleOrRights(UserIdentity user, String[] rights, Object context)
    {
        for (String right : rights)
        {
            if (hasRight(user, right, context))
            {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Test if a user has a right on a context.
     * @param user the user
     * @param right the right to check.
     * @param context the right context.
     * @return true if the user has the right, false otherwise.
     */
    protected boolean hasRight(UserIdentity user, String right, Object context)
    {
        return _rightManager.hasRight(user, right, context) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Compute the context to use.<br>
     * Default implementation uses standard context <code>"/cms"</code>.
     * @param transientVars variables that will not be persisted.
     * @param args the properties for this function invocation.
     * @param user the current user.
     * @param right the needed right.
     * @return the computed context.
     * @throws WorkflowException if an error occurs.
     */
    protected Object _computeContext(Map transientVars, Map args, UserIdentity user, String right) throws WorkflowException
    {
        // Retrieve the context to use
        String context = (String) args.get(__CONTEXT_KEY);

        if (context == null)
        {
            if (_logger.isDebugEnabled())
            {
                _logger.debug(String.format("Missing 'context' argument, using default context: %s",
                                            __DEFAULT_CONTEXT));
            }
            
            return __DEFAULT_CONTEXT;
        }
        
        return context;
    }

    @Override
    public void dispose()
    {
        _manager.release(_rightManager);
        _rightManager = null;
        _manager = null;
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<WorkflowArgument> getArguments()
    {
        WorkflowArgument rights = WorkflowElementDefinitionHelper.getElementDefinition(
            __RIGHT_KEY,
            new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_CHECK_RIGHTS_ARGUMENT_RIGHT_KEY_LABEL"),
            new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_CHECK_RIGHTS_ARGUMENT_RIGHT_KEY_DESCRIPTION"),
            true,
            false
        );
        rights.setEnumerator(_getRightsEnumerator());
        
        return List.of(
                rights,
                WorkflowElementDefinitionHelper.getElementDefinition(
                        __CONTEXT_KEY,
                        new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_CHECK_RIGHTS_ARGUMENT_CONTEXT_KEY_LABEL"),
                        new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_CHECK_RIGHTS_ARGUMENT_CONTEXT_KEY_DESCRIPTION"),
                        false,
                        false
                )
        );
    }
    
    /**
     * Get the rights enumerator
     * @return the rights enumerator
     */
    protected StaticEnumerator _getRightsEnumerator()
    {
        StaticEnumerator<String> rightsStaticEnumerator = new StaticEnumerator<>();
        for (String rightId : _rightsExtensionPoint.getExtensionsIds())
        {
            Right right = _rightsExtensionPoint.getExtension(rightId);
            Map<String, I18nizableTextParameter> params = Map.of("category", right.getCategory(), "label", right.getLabel());
            rightsStaticEnumerator.add(new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_CHECK_RIGHTS_ARGUMENT_RIGHT_KEY_PARAMS_LABEL", params), right.getId());
        }
        
        return rightsStaticEnumerator;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_EDITOR_CHECK_RIGHTS_CONDITION_LABEL");
    }
    
    /**
     * Get condition description when there are multiple rights involved
     * @param parameters list of rights
     * @return the description
     */
    protected I18nizableText _getMultipleConditionsDescriptionKey(List<String> parameters)
    {
        return new I18nizableText("plugin.workflow", "UITOOL_WORKFLOW_EDITOR_CHECK_RIGHTS_MULTIPLES_CONDITION_DESCRIPTION", parameters);
    }
    
    /**
     * Get condition description when there is one right
     * @param parameters the right label as a List
     * @return the description
     */
    protected I18nizableText _getSingleConditionDescriptionKey(List<String> parameters)
    {
        return new I18nizableText("plugin.workflow", "UITOOL_WORKFLOW_EDITOR_CHECK_RIGHTS_CONDITION_DESCRIPTION", parameters);
    }
    
    @Override
    public I18nizableText getFullLabel(Map<String, String> argumentsValues)
    {
        String rightNeeded = argumentsValues.get(__RIGHT_KEY);
        String translatedRight = "";
        if (rightNeeded.contains("|"))
        {
            String[] rightsOr = StringUtils.split(rightNeeded, '|');
            translatedRight = "<strong>" + _i18nUtils.translate(_rightsExtensionPoint.getExtension(rightsOr[0]).getLabel()) + "</strong>";
            for (String rightId: rightsOr)
            {
                Right right = _rightsExtensionPoint.getExtension(rightId);
                translatedRight += _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_CONDITION_OR")) + "<strong>" + _i18nUtils.translate(right.getDescription()) + "</strong>";
            }
            return _getMultipleConditionsDescriptionKey(List.of(translatedRight));
        }
        else if (rightNeeded.contains("&"))
        {
            String[] rightsAnd = StringUtils.split(rightNeeded, '&');
            translatedRight = "<strong>" + _i18nUtils.translate(_rightsExtensionPoint.getExtension(rightsAnd[0]).getLabel()) + "</strong>";
            for (String rightId: rightsAnd)
            {
                Right right = _rightsExtensionPoint.getExtension(rightId);
                translatedRight += _i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGINS_WORKFLOW_EDITOR_CONDITION_AND")) + "<strong>" + _i18nUtils.translate(right.getDescription()) + "</strong>";
            }
            return _getMultipleConditionsDescriptionKey(List.of(translatedRight));
        }
        Right right = _rightsExtensionPoint.getExtension(rightNeeded);
        translatedRight = "<strong>" + _i18nUtils.translate(right.getLabel()) + "</strong>";
        return _getSingleConditionDescriptionKey(List.of(translatedRight));
    }
}
