/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.loader.AbstractDescriptor;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.ConditionalResultDescriptor;
import com.opensymphony.workflow.loader.ConditionsDescriptor;
import com.opensymphony.workflow.loader.DescriptorFactory;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * DAO for workflow results
 */
public class WorkflowResultDAO extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role */ 
    public static final String ROLE = WorkflowResultDAO.class.getName();
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The workflow step DAO */
    protected WorkflowStepDAO _workflowStepDAO;
    
    /** The workflow transition DAO */
    protected WorkflowTransitionDAO _workflowTransitionDAO;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _workflowSessionHelper = (WorkflowSessionHelper) manager.lookup(WorkflowSessionHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) manager.lookup(WorflowRightHelper.ROLE);
        _workflowStepDAO = (WorkflowStepDAO) manager.lookup(WorkflowStepDAO.ROLE);
        _workflowTransitionDAO = (WorkflowTransitionDAO) manager.lookup(WorkflowTransitionDAO.ROLE);
    }
    
    /**
     * Get a list of the action's conditional results in a json format 
     * @param workflowName  the current workflow's  unique name
     * @param actionId  the current action
     * @param currentStepId in edition mode, id of the selected final step 
     * @return a list of available conditional results for this action
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getConditionalResultsToJson(String workflowName, Integer actionId, Integer currentStepId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkReadRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        List<Integer> usedSteps = new ArrayList<>();
        usedSteps.add(action.getUnconditionalResult().getStep());
        List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
        for (ConditionalResultDescriptor result : conditionalResults)
        {
            usedSteps.add(result.getStep());
        }
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> workflowStatesToJson = (List<Map<String, Object>>) _workflowStepDAO.getStatesToJson(workflowName, actionId, workflowDescriptor.getInitialAction(actionId) != null).get("data");
        List<Map<String, Object>> states = new ArrayList<>();
        for (int i = 0; i < workflowStatesToJson.size(); i++)
        {
            Map<String, Object> state = workflowStatesToJson.get(i);
            Integer id = (Integer) state.get("id");
            if (!usedSteps.contains(id))
            {
                states.add(state);
            }
        }
        if (currentStepId != null)
        {
            Map<String, Object> stateInfos = new HashMap<>();
            stateInfos.put("id", currentStepId);
            boolean showId = currentStepId != -1;
            stateInfos.put("label", _workflowStepDAO.getStepLabelAsString(workflowDescriptor, currentStepId, showId));
            states.add(stateInfos);
        }
        
        return Map.of("data", states);
    }
    
    /**
     * Add a conditional result to action
     * @param workflowName unique name of current workflow
     * @param stepId id of the parent step of the selected action
     * @param actionId id of selected action
     * @param resultId id of step to add as conditional result
     * @return map of result's infos
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addConditionalResult(String workflowName, Integer stepId, Integer actionId, Integer resultId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
        Optional<ConditionalResultDescriptor> sameFinalState = conditionalResults.stream().filter(cr -> cr.getStep() == resultId).findAny();
        if (sameFinalState.isPresent())
        {
            return Map.of("message", "duplicate-state");
        }
        
        DescriptorFactory factory = new DescriptorFactory();
        ConditionalResultDescriptor resultDescriptor = factory.createConditionalResultDescriptor();
        resultDescriptor.setStep(resultId);
        conditionalResults.add(resultDescriptor);
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getResultProperties(workflowDescriptor, action, stepId, resultId);
    }
    
    /**
     * Edit a conditional result
     * @param workflowName unique name of current workflow
     * @param stepId id of the parent step of the selected action
     * @param actionId id of selected action
     * @param resultOldId former id of selected result
     * @param resultNewId new id for selected result
     * @return map of result's infos
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editConditionalResult(String workflowName, Integer stepId, Integer actionId, Integer resultOldId, Integer resultNewId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
        int indexOf = getIndexOfStepResult(conditionalResults, resultOldId);
        if (indexOf != -1)
        {
            conditionalResults.get(indexOf).setStep(resultNewId);
        }
        
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getResultProperties(workflowDescriptor, action, stepId, resultNewId);
    }
    
    /**
     * Delete a conditional result from action
     * @param workflowName unique name of current workflow
     * @param stepId id of the parent step of the selected action
     * @param actionId id of selected action
     * @param resultId id of the result to delete
     * @return map of result's infos
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteConditionalResult(String workflowName, Integer stepId, Integer actionId, Integer resultId)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, true);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        ActionDescriptor action = workflowDescriptor.getAction(actionId);
        List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
        int indexOf = getIndexOfStepResult(conditionalResults, resultId);
        if (indexOf != -1)
        {
            conditionalResults.remove(indexOf);
        }
        _workflowSessionHelper.updateWorkflowDescriptor(workflowDescriptor);
        
        return _getResultProperties(workflowDescriptor, action, stepId, resultId);
    }

    private Map<String, Object> _getResultProperties(WorkflowDescriptor workflowDescriptor, ActionDescriptor action, Integer stepId, Integer resultId)
    {
        Map<String, Object> results = new HashMap<>();
        results.put("nodeId", "step" + resultId);
        results.put("nodeLabel", _workflowStepDAO.getStepLabel(workflowDescriptor, resultId));
        results.put("actionId", action.getId());
        results.put("actionLabel", _workflowTransitionDAO.getActionLabel(action));
        results.put("stepId", stepId);
        results.put("stepLabel", _workflowStepDAO.getStepLabel(workflowDescriptor, stepId));
        results.put("workflowId", workflowDescriptor.getName());
        results.put("isConditional", true);
        return results;
    }
    
    /**
     * Get children conditions of current node in result tree
     * @param currentNode id of current node in tree
     * @param action the selected action
     * @param conditionalResults the action's conditional results
     * @return a list of conditions, can be empty
     */
    protected List<AbstractDescriptor> getChildrenResultConditions(String currentNode, ActionDescriptor action, List<ConditionalResultDescriptor> conditionalResults)
    {
        if (!conditionalResults.isEmpty())
        {
            String[] path = getPath(currentNode);
            int stepId = Integer.valueOf(path[0].substring(4));
            List<ConditionsDescriptor> resultConditions = getRootResultConditions(conditionalResults, stepId);
            if (!resultConditions.isEmpty())
            {
                ConditionsDescriptor rootConditionsDescriptor = resultConditions.get(0);
                
                // The current node is root and it's a OR node, so display it ...
                if (path.length == 1 && rootConditionsDescriptor.getType().equals(WorkflowConditionDAO.OR))
                {
                    return List.of(rootConditionsDescriptor);
                }
                // ... the current node is a AND node, display child conditions ...
                else if (path.length == 1)
                {
                    return rootConditionsDescriptor.getConditions();
                }
                // ... the selected node is not a condition, so it has children
                // we need to search the condition and get child condition of current node
                else if (!path[path.length - 1].startsWith("condition")) 
                {
                    List<AbstractDescriptor> conditions = new ArrayList<>();
                    conditions.add(rootConditionsDescriptor);
                    if (!conditions.isEmpty())
                    {
                        // get conditions for current node
                        int i = 1;
                        do
                        {
                            String currentOrAndConditionId = path[i];
                            int currentOrAndConditionIndex = (currentOrAndConditionId.startsWith("and"))
                                    ? Integer.valueOf(currentOrAndConditionId.substring(3))
                                    : Integer.valueOf(currentOrAndConditionId.substring(2));
                            
                            ConditionsDescriptor currentOrAndCondition = (ConditionsDescriptor) conditions.get(currentOrAndConditionIndex);
                            conditions = currentOrAndCondition.getConditions();
                            i++;
                        }
                        while (i < path.length);
                        
                        return conditions;
                    }
                }
            }
        }
        
        return List.of();
    }

    /**
     * Get the path to the current node
     * @param currentNodeId id of the current node, is like 'step1-(and|or)0-condition4'
     * @return an array of the path (step1, and0, condition4)
     */
    public String[] getPath(String currentNodeId)
    {
        String[] path = currentNodeId.split("-");
        if (path.length > 1 && path[1].equals("1"))
        {
            path[0] += "-1";
            String[] copyArray = new String[path.length - 1]; 
            copyArray[0] = path[0];
            System.arraycopy(path, 2, copyArray, 1, path.length - 2); 
            path = copyArray;
        }
        return path;
    }
    
    /**
     * Get the root ConditionsDescriptor having current step as result
     * this is the conditions descriptor that decide whether children conditions are linked by an AND or an OR operator
     * @param conditionalResults list of current action's conditional results
     * @param stepId id of current targeted step
     * @return a list of the root ConditionsDescriptor if exist
     */
    public List<ConditionsDescriptor> getRootResultConditions(List<ConditionalResultDescriptor> conditionalResults, int stepId)
    {
        return conditionalResults.stream()
                    .filter(r -> r.getStep() == stepId)
                    .map(r -> r.getConditions())
                    .findFirst()
                    .orElse(List.of());
    }
    
    /**
     * Get the index of selected step result in the list of conditional results
     * @param conditionalResults a list of conditional results
     * @param stepResultId id of the result to find
     * @return the step result's index
     */
    public int getIndexOfStepResult(List<ConditionalResultDescriptor> conditionalResults, Integer stepResultId)
    {
        for (int i = 0; i < conditionalResults.size(); i++)
        {
            if (conditionalResults.get(i).getStep() == stepResultId)
            {
                return i;
            }
        }
        return -1;
    }
}
