/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The workflow property DAO
 */
Ext.define('Ametys.plugins.workflow.dao.WorkflowPropertyDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowPropertyDAO
         * @method addProperty 
         * Add workflow property
         * This calls the method 'addProperty' of the server DAO 'WorkflowPropertyDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowPropertyDAO",
                methodName: "addProperty",
                callback: {
                    handler: this._addPropertyCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_PROPERTY_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowPropertyDAO
         * @method editProperty 
         * Edit workflow property
         * This calls the method 'editProperty' of the server DAO 'WorkflowPropertyDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowPropertyDAO",
                methodName: "editProperty",
                callback: {
                    handler: this._editPropertyCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_PROPERTY_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowPropertyDAO
         * @method deleteProperty 
         * Delete workflow property
         * This calls the method 'deleteProperty' of the server DAO 'WorkflowPropertyDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowPropertyDAO",
                methodName: "deleteProperty",
                callback: {
                    handler: this._deletePropertyCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_PROPERTY_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowPropertyDAO
         * @method getPropertiesNames 
         * Get workflow element properties names
         * This calls the method 'getPropertiesNames' of the server DAO 'WorkflowPropertyDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowPropertyDAO",
                methodName: "getPropertiesNames",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_PROPERTY_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * Callback function called after {@link #addProperty} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _addPropertyCB: function(response)
    {
        if (response.message == 'blank-name')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_ADD_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_BLANK_NAME_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else if (response.message == 'duplicate-name')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_ADD_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_DUPLICATE_NAME_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {
            var propertyTarget = this._getPropertyTarget(response)
           
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.CREATED,
                targets: [propertyTarget]
            });
            
            this._sendWorkflowModifiedMessage(response);
        }
        
    },
    
    /**
     * Callback function called after {@link #editProperty} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _editPropertyCB: function(response)
    {
        if (response.message == 'blank-name')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_EDIT_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_BLANK_NAME_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        
        var propertyTarget = this._getPropertyTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [propertyTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * @private
     * Callback of _doRemove, send bus message
     * @param {Object} response the server response
     * @param {String} response.message the error message, can be undefined
     */
    _deletePropertyCB: function(response)
    {
        if (response.message == 'blank-name')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_WORKFLOW_DELETE_ERROR_TITLE}}",
                msg: "{{i18n PLUGINS_WORKFLOW_BLANK_NAME_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        
        var propertyTarget = this._getPropertyTarget(response);
        
        Ext.create('Ametys.message.Message', {
            type: Ametys.message.Message.DELETED,
            targets: [propertyTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * @private
     * Get property target
     * @param {Object} response the server response
     * @param {Number} response.stepId the parent step id
     * @param {Number} response.actionId the parent action id, can be null if direct parent is a step
     * @param {String} response.name the property's name 
     * @param {String} response.value the property's value
     * @returns {Object} the property's target 
     */
    _getPropertyTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_PROPERTY,
            parameters: {
                name: response.name,
                value: response.value,
                workflowId: response.workflowId,
                stepId: response.stepId,
                actionId: response.actionId
            }
        };
    },
    
   /**
     * @private
     * Get the action target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the parent step
     * @param {String} response.actionId the id of the action
     * @param {String} response.actionLabel the label of the action
     */
    _getActionTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_ACTION,
            parameters: {
                id: response.actionId,
                label:response.actionLabel,
                stepId: response.stepId,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Get the step target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the step
     * @param {String} response.stepLabel the label of the step
     */
    _getStepTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_STEP,
            parameters: {
                id: response.stepId,
                label:response.stepLabel,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Send a message modified for workflow target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     */
    _sendWorkflowModifiedMessage: function(response)
    {
        var workflowTarget = {
            id: Ametys.message.MessageTarget.WORKFLOW_OBJECT,
            parameters: {
                id: response.workflowId
            }
        };
        var stepTarget = this._getStepTarget(response);
        if (response.actionId != null)
        {
            stepTarget.subtargets = this._getActionTarget(response);
        }
        workflowTarget.subtargets = stepTarget;
            
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [workflowTarget]
        });
    }
});
