/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.dao;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.environment.Context;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.workflow.ObservationConstants;
import org.ametys.plugins.workflow.component.WorkflowLanguageManager;
import org.ametys.plugins.workflow.definition.WorkflowDefinitionExtensionPoint;
import org.ametys.plugins.workflow.support.I18nHelper;
import org.ametys.plugins.workflow.support.WorflowRightHelper;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowSessionHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.loader.AbstractDescriptor;
import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.ConditionalResultDescriptor;
import com.opensymphony.workflow.loader.ConditionsDescriptor;
import com.opensymphony.workflow.loader.DescriptorFactory;
import com.opensymphony.workflow.loader.RestrictionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * DAO for managing workflows
 */
public class WorkflowsDAO extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = WorkflowsDAO.class.getName();
    /** Meta to set in new workflows */
    public static final String META_NEW_WORKFLOW = "new-workflow";
    
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    
    /** The workflow session helper */
    protected WorkflowSessionHelper _workflowSessionHelper;
    
    /** The workflow right helper */
    protected WorflowRightHelper _workflowRightHelper;
    
    /** The workflow language manager */
    protected WorkflowLanguageManager _workflowLanguageManager;
    
    /** The helper for i18n translations and catalogs */
    protected I18nHelper _i18nHelper;
    
    /** The Cocoon context */
    protected Context _cocoonContext;
    
    /** I18n Utils */
    protected I18nUtils _i18nUtils;
    
    /** The context */
    protected org.apache.avalon.framework.context.Context _context;
    
    /** The regex pattern for workflow names */
    protected final String _regexPattern = "^[a-zA-Z\\-]+$";
    
    /** The Workflow Definition Extension Point */
    protected WorkflowDefinitionExtensionPoint _workflowDefinitionEP;
    
    /** The workflow transition DAO */
    protected WorkflowTransitionDAO _workflowTransitionDAO;
    
    /** The observation manager */
    protected ObservationManager _observationManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The source resolver */
    protected SourceResolver _sourceResolver;
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
        _workflowDefinitionEP = (WorkflowDefinitionExtensionPoint) manager.lookup(WorkflowDefinitionExtensionPoint.ROLE);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
        _workflowRightHelper = (WorflowRightHelper) manager.lookup(WorflowRightHelper.ROLE);
        _workflowSessionHelper = (WorkflowSessionHelper) manager.lookup(WorkflowSessionHelper.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _workflowLanguageManager = (WorkflowLanguageManager) manager.lookup(WorkflowLanguageManager.ROLE);
        _i18nHelper = (I18nHelper) manager.lookup(I18nHelper.ROLE);
        _workflowTransitionDAO = (WorkflowTransitionDAO) manager.lookup(WorkflowTransitionDAO.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Get the workflow properties
     * @param workflowName the name of the workflow to get
     * @return the workflow properties
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getWorkflowRootProperties(String workflowName)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        Map<String, Object> infos = new HashMap<>();
        if (workflowDescriptor != null && _workflowRightHelper.canRead(workflowDescriptor))
        {
            infos.put("id", workflowName);
            infos.put("label", _workflowSessionHelper.getWorkflowLabel(workflowName));
            infos.put("hasChildren", workflowDescriptor.getSteps().size() > 0);
            infos.put("hasChanges", _workflowSessionHelper.hasChanges(workflowName));
            infos.put("isNew", workflowDescriptor.getMetaAttributes().containsKey(META_NEW_WORKFLOW));
            infos.put("canWrite", _workflowRightHelper.canWrite(workflowDescriptor));
        }
        else
        {
            String errorMsg = workflowDescriptor != null ? "cant-read" : "workflow-unknown";
            infos.put("error", errorMsg);
        }
        return infos;
    }
    
    /**
     * Get the list of all workflows
     * @return a map with workflow's list as value
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getWorkflowsList()
    {
        List<Map<String, Object>> workflows2json = new ArrayList<>();
        
        Set<String> workflowNames = _workflowSessionHelper.getWorkflowNames();
        for (String workflowName: workflowNames)
        {
            WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
            if (_workflowRightHelper.canRead(workflowDescriptor))
            {
                Map<String, Object> workflowData = new LinkedHashMap<>();
                workflowData.put("title", _workflowSessionHelper.getWorkflowLabel(workflowName));
                workflowData.put("id", workflowName);
                workflowData.put("hasChanges", _workflowSessionHelper.hasChanges(workflowName));
                workflows2json.add(workflowData);
            }
        }
        
        return Map.of("workflows", workflows2json);
    }
    
    /**
     * Overwrite the current workflow in a XML file  
     * @param workflowName id of current workflow
     * @return an empty map if all went well, an error message if not  
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> saveChanges(String workflowName)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        Map<String, Object> results = new HashMap<>();
        
        try
        {
            // Write new workflow
            if (!_setErrors(workflowDescriptor, results))
            {
                workflowDescriptor.getMetaAttributes().remove(META_NEW_WORKFLOW);
                _writeWorklowFile(workflowName, workflowDescriptor);
                
                // Write new i18n
                _writeI18nTranslations(workflowName, _workflowSessionHelper.getTranslations(workflowName));
                
                _workflowSessionHelper.cloneImages(workflowName);
                _workflowSessionHelper.deleteSession(workflowName);
                _workflowDefinitionEP.addOrUpdateExtension(workflowName);
                
                _i18nHelper.clearCaches();
                
                Map<String, Object> params = new HashMap<>();
                params.put(ObservationConstants.ARGS_WORKFLOW_NAME, workflowName);
                _observationManager.notify(new Event(ObservationConstants.EVENT_WORKFLOW_SAVED, _currentUserProvider.getUser(), params));
            }
            results.put("workflowId", workflowName);
        }
        catch (FileNotFoundException e)
        {
            results.put("message", "file-not-found");
            getLogger().error("An error occured while overwriting workflow file: {}", workflowName, e);
        }
        catch (Exception e)
        {
            results.put("message", "sax-error");
            getLogger().error("An error occured while saxing i18n catalogs file", e);
        }
        return results;
    }
    
    /**
     * Restore last version of current workflow if exist
     * @param workflowName name of current workflow
     * @return map of result
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> reinit(String workflowName)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        if (workflowDescriptor.getMetaAttributes().containsKey(META_NEW_WORKFLOW))
        {
            return Map.of("message", "new_workflow");
        }
        _workflowSessionHelper.deleteSession(workflowName);
        return Map.of("workflowId", workflowName);
    }
    
    /**
     * Check for invalid components in workflow, return true if there is any
     * @param workflowDescriptor the workflow to check
     * @param results a map to fill with error message and invalid component's labels
     * @return true if there are errors
     */
    protected boolean _setErrors(WorkflowDescriptor workflowDescriptor, Map<String, Object> results)
    {
        if (workflowDescriptor.getInitialActions().isEmpty())
        {
            results.put("message", "empty-initials-actions");
            return true;
        }
        Set<Integer> transitionIds = _workflowHelper.getAllActions(workflowDescriptor);
        return _hasEmptyOperator(workflowDescriptor, transitionIds, results) || _hasEmptyConditionalResult(workflowDescriptor, results, transitionIds);
    }
    
    /**
     * Check for result without condition or operators without conditions
     * @param workflowDescriptor the workflow to check
     * @param results a map to fill with error message and invalid transition's labels
     * @param transitionIds a list of all the workflow's transitions ids
     * @return true if there are invalid conditional results
     */
    protected boolean _hasEmptyConditionalResult(WorkflowDescriptor workflowDescriptor, Map<String, Object> results, Set<Integer> transitionIds)
    {
        Set<String> invalidTransitions = new HashSet<>();
        for (Integer id : transitionIds)
        {
            ActionDescriptor action = workflowDescriptor.getAction(id);
            String actionLabel = _workflowTransitionDAO.getActionLabel(workflowDescriptor.getName(), action) + "(" + action.getId() + ")";
            
            List<ConditionalResultDescriptor> conditionalResults = action.getConditionalResults();
            for (ConditionalResultDescriptor resultDescriptor : conditionalResults)
            {
                List<ConditionsDescriptor> conditions = resultDescriptor.getConditions();
                if (conditions.isEmpty())
                {
                    results.put("message", "empty-conditionnal-result");
                    invalidTransitions.add(actionLabel);
                }
                else if (_hasOperatorWithoutChild(conditions.get(0)))
                {
                    results.put("message", "empty-result-operator");
                    invalidTransitions.add(actionLabel);
                }
            }
        }
        
        results.put("invalidTransitions", invalidTransitions.isEmpty() ? null : invalidTransitions);
        return results.get("message") != null;
    }
    
    /**
     * Check for operators without conditions
     * @param workflowDescriptor the workflow to check
     * @param transitionIds a list of all the workflow's transitions ids
     * @param results a map to fill with error message and invalid transition's labels
     * @return true if there are operator without conditions
     */
    protected boolean _hasEmptyOperator(WorkflowDescriptor workflowDescriptor, Set<Integer> transitionIds, Map<String, Object> results)
    {
        Set<String> invalidTransitions = new HashSet<>();
        for (Integer id : transitionIds)
        {
            ActionDescriptor action = workflowDescriptor.getAction(id);
            if (action != null) // case in kernel workflow where an action is declared but never used
            {
                String actionLabel = _i18nHelper.translateKey(workflowDescriptor.getName(), new I18nizableText("application", action.getName()), WorkflowTransitionDAO.DEFAULT_ACTION_NAME) + " (" + action.getId() + ")";
                RestrictionDescriptor restriction = action.getRestriction();
                if (restriction != null)
                {
                    ConditionsDescriptor rootOperator = restriction.getConditionsDescriptor();
                    if (_hasOperatorWithoutChild(rootOperator))
                    {
                        invalidTransitions.add(actionLabel);
                    }
                }
            }
        }
        if (!invalidTransitions.isEmpty())
        {
            results.put("message", "empty-condition-operator");
            results.put("invalidTransitions", invalidTransitions);
            return true;
        }
        return false;
    }

    private boolean _hasOperatorWithoutChild(ConditionsDescriptor rootOperator)
    {
        List<AbstractDescriptor> conditions = rootOperator.getConditions();
        if (conditions.isEmpty())
        {
            return true;
        }
        else
        {
            int i = 0;
            boolean hasChildFreeOperator = false;
            while (i < conditions.size() && !hasChildFreeOperator)
            {
                if (conditions.get(i) instanceof ConditionsDescriptor operator)
                {
                    hasChildFreeOperator = _hasOperatorWithoutChild(operator);
                }
                i++;
            }
            return hasChildFreeOperator;
        }
    }
    
    private void _writeI18nTranslations(String workflowName, Map<I18nizableText, Map<String, String>> translations) throws Exception
    {
        if (!translations.isEmpty())
        {
            // Write new i18n messages in application 
            String workflowCatalog = _workflowHelper.getWorkflowCatalog(workflowName);
            Map<String, Map<I18nizableText, String>> newI18nCatalogs = _i18nHelper.createNewI18nCatalogs(translations);
            _i18nHelper.saveCatalogs(newI18nCatalogs, workflowCatalog);
        }
    }
    
    private void _writeWorklowFile(String workflowName, WorkflowDescriptor workflowDescriptor) throws IOException
    {
        File workflowFile = new File(_workflowHelper.getParamWorkflowDir(), workflowName + ".xml");
        
        // Save the workflow file if it already exists
        if (workflowFile.exists())
        {
            FileUtils.copyFile(workflowFile, new File(workflowFile + ".bak"), StandardCopyOption.REPLACE_EXISTING);
        }
        // Otherwise, create the file and its parents if necessary
        else
        {
            FileUtils.createParentDirectories(workflowFile);
            workflowFile.createNewFile();
        }
        
        try (PrintWriter out = new PrintWriter(workflowFile, StandardCharsets.UTF_8))
        {
            out.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
            out.println("<!DOCTYPE workflow PUBLIC \"-//OpenSymphony Group//DTD OSWorkflow 2.8//EN\" \"http://www.opensymphony.com/osworkflow/workflow_2_8.dtd\">");
            workflowDescriptor.writeXML(out, 0);
        }
    }

    /**
     * Get multilingual labels for current workflow
     * @param workflowName name of current workflow
     * @return a map of labels, key is language and value is translation
     */
    protected Map<String, String> _getWorkflowLabels(String workflowName)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        Map<String, String> workflowLabelTranslations = _workflowSessionHelper.getWorkflowLabelTranslations(workflowName);
        if (workflowLabelTranslations.isEmpty() && ArrayUtils.contains(_workflowHelper.getWorkflowNames(), workflowName))
        {
            I18nizableText workflowI18nLabelKey = _workflowHelper.getWorkflowLabel(workflowName);
            for (String language : _workflowLanguageManager.getLanguages())
            {
                workflowLabelTranslations.put(language, StringUtils.defaultString(_i18nUtils.translate(workflowI18nLabelKey, language)));
            }
        }
        return workflowLabelTranslations;
    }
    
    /**
     * Get workflow infos 
     * @param workflowName the name of the  workflow
     * @return a map of the list of workflow names and the workflow's labels
     */
    @Callable(rights = {"Workflow_Right_Edit", "Workflow_Right_Edit_User"})
    public Map<String, Object> getWorkflowInfos(String workflowName)
    {
        Map<String, Object> workflowInfos = new HashMap<>();
        workflowInfos.put("workflowNames",  _workflowSessionHelper.getWorkflowNames());
        if (StringUtils.isNotBlank(workflowName))
        {
            workflowInfos.put("labels", _getWorkflowLabels(workflowName));
        }
        return workflowInfos;
    }
    
    /**
     * Create a new workflow 
     * @param labels the multilingual labels
     * @param id the unique name 
     * @return map of error message or workflow name if creation went well
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createWorkflow(Map<String, String> labels, String id)
    {
        // Check user right
        _workflowRightHelper.checkEditRight();
        
        Set<String> workflowNames = _workflowSessionHelper.getWorkflowNames();
        if (workflowNames.contains(id))
        {
            return Map.of("message", "duplicate-id");
        }
        
        //create workflow descriptor
        DescriptorFactory factory = new DescriptorFactory();
        WorkflowDescriptor workflowDescriptor = factory.createWorkflowDescriptor();
        workflowDescriptor.setName(id);
        workflowDescriptor.getMetaAttributes().put("user", true);
        
        //add meta to new workflow to prevent reinit
        workflowDescriptor.getMetaAttributes().put(META_NEW_WORKFLOW, true);
        
        //save workflow in session
        _workflowSessionHelper.initWorkflowDescriptor(workflowDescriptor);
        _workflowSessionHelper.updateWorkflowNames(workflowDescriptor);
        
        //add workflow label translations
        _workflowSessionHelper.updateTranslations(id, _i18nHelper.getWorkflowLabelKey(id), labels);
        
        return Map.of("workflowId", id);
    }
    
    /**
     * Rename the workflow
     * @param workflowName unique name of current workflow
     * @param labels the new multilingual labels
     * @return the workflow name
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> renameWorkflow(String workflowName, Map<String, String> labels)
    {
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(workflowName, false);
        
        // Check user right
        _workflowRightHelper.checkEditRight(workflowDescriptor);
        
        // Create i18n entry
        I18nizableText labelKey = _i18nHelper.getWorkflowLabelKey(workflowName);
        _workflowSessionHelper.updateTranslations(workflowName, labelKey, labels);
        return Map.of("workflowId", workflowName);
    }
    
    /**
     * Duplicate a workflow
     * @param newWorkflowName the new workflow name
     * @param labels the new labels for the workflow
     * @param duplicatedWorkflowName the duplicated workflow name
     * @return map of results
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> duplicateWorkflow(String newWorkflowName, Map<String, String> labels, String duplicatedWorkflowName)
    {
        // Check user right
        _workflowRightHelper.checkEditRight();
        
        Set<String> workflowNames = _workflowSessionHelper.getWorkflowNames();
        if (workflowNames.contains(newWorkflowName))
        {
            return Map.of("message", "duplicate-id");
        }
        
        WorkflowDescriptor workflowDescriptor = _workflowSessionHelper.getWorkflowDescriptor(duplicatedWorkflowName, false);
        if (workflowDescriptor.getMetaAttributes().containsKey(META_NEW_WORKFLOW))
        {
            return Map.of("message", "no-save");
        }
        
        _workflowSessionHelper.duplicateWorkflow(newWorkflowName, labels, duplicatedWorkflowName);
        
        return Map.of("workflowId", newWorkflowName);
    }

}
