/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The workflow condition DAO
 */
Ext.define('Ametys.plugins.workflow.dao.WorkflowConditionDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method addOperator 
         * Add workflow conditions
         * This calls the method 'addOperator' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "addOperator",
                callback: {
                    handler: this._addOperatorCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method deleteOperator 
         * Add workflow conditions
         * This calls the method 'addOperator' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "deleteOperator",
                callback: {
                    handler: this._deleteOperatorCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method getConditionsModel 
         * Get arguments for workflow condition
         * This calls the method 'getConditionsModel' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "getConditionsModel",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: true,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method addCondition 
         * Add a condition to action
         * This calls the method 'addCondition' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "addCondition",
                callback: {
                    handler: this._addConditionCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method getConditionParametersValues 
         * Get the condition's parameters and their values 
         * This calls the method 'getConditionParametersValues' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "getConditionParametersValues",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method editCondition 
         * Edit a condition
         * This calls the method 'editCondition' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "editCondition",
                callback: {
                    handler: this._editConditionCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method deleteCondition 
         * Add a condition to action
         * This calls the method 'deleteCondition' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "deleteCondition",
                callback: {
                    handler: this._deleteConditionCB
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method hasChildConditions 
         * Return true if current operator has child conditions
         * This calls the method 'hasChildConditions' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "hasChildConditions",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.workflow.dao.WorkflowConditionDAO
         * @method getConditionLabel 
         * Get the label of a condition
         * This calls the method 'getConditionLabel' of the server DAO 'WorkflowConditionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.workflow.dao.WorkflowConditionDAO",
                methodName: "getConditionLabel",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n PLUGINS_WORKFLOW_CONDITION_DAO_ERROR_TITLE}}",
                    category: Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * Callback function called after {@link #addOperator} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _addOperatorCB: function(response)
    {
        var conditionsTarget = this._getConditionOperatorTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: [conditionsTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #deleteOperator} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _deleteOperatorCB: function(response)
    {
        var conditionsTarget = this._getConditionOperatorTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.DELETED,
            targets: [conditionsTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #editCondition} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _editConditionCB: function(response)
    {
        var conditionsTarget = this._getConditionTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [conditionsTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #deleteCondition} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _deleteConditionCB: function(response)
    {
        var conditionsTarget = this._getConditionTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.DELETED,
            targets: [conditionsTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * Callback function called after {@link #addCondition} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _addConditionCB: function(response)
    {
        var conditionTarget = this._getConditionTarget(response)
       
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: [conditionTarget]
        });
        
        this._sendWorkflowModifiedMessage(response);
    },
    
    /**
     * @private
     * Get the conditions operator target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.actionId the id of the action
     * @param {String} response.nodeId the id of conditions node in its tree panel
     * @param {String} response.type the type of operator can be AND or OR
     */
    _getConditionOperatorTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_CONDITIONS_OPERATOR,
            parameters: {
                nodeId: response.nodeId,
                type: response.type,
                actionId: response.actionId,
                stepId: response.stepId,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Get the conditions target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.actionId the id of the action
     * @param {String} response.nodeId the id of condition node in its tree panel
     */
    _getConditionTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_CONDITION,
            parameters: {
                nodeId: response.nodeId,
                conditionId: response.conditionId,
                actionId: response.actionId,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Get the action target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the parent step
     * @param {String} response.actionId the id of the action
     */
    _getActionTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_ACTION,
            parameters: {
                id: response.actionId,
                label: response.actionLabel,
                stepId: response.stepId,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Get the step target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     * @param {String} response.stepId the id of the step
     */
    _getStepTarget: function(response)
    {
        return {
            id: Ametys.message.MessageTarget.WORKFLOW_STEP,
            parameters: {
                id: response.stepId,
                label:response.stepLabel,
                workflowId: response.workflowId
            }
        };
    },
    
    /**
     * @private
     * Send a message modified for workflow target
     * @param {Object} response the serveur response
     * @param {String} response.workflowId the name of current workflow
     */
    _sendWorkflowModifiedMessage: function(response)
    {
        var workflowTarget = {
            id: Ametys.message.MessageTarget.WORKFLOW_OBJECT,
            parameters: {
                id: response.workflowId
            }
        };
        var stepTarget = this._getStepTarget(response);
        stepTarget.subtargets = this._getActionTarget(response);
        workflowTarget.subtargets = stepTarget;
            
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: [workflowTarget]
        });
    }
    
});
