/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.parameter.ValidationResult;

import com.opensymphony.workflow.WorkflowContext;
import com.opensymphony.workflow.WorkflowException;

/**
 * Abstract class for easily retrieving environment components
 * in a condition or a function.
 */
public abstract class AbstractWorkflowComponent implements LogEnabled, Serviceable
{
    /** Constant for storing the content into the transient variables map. */
    public static final String CONTEXT_PARAMETERS_KEY = AbstractWorkflowComponent.class.getName() + "$Parameters";
    
    /** Constant for storing the result map into the transient variables map. */
    public static final String RESULT_MAP_KEY = AbstractWorkflowComponent.class.getName() + ";resultMap";
    
    /** Constant for storing the content into the transient variables map. */
    public static final String FAIL_CONDITIONS_KEY = AbstractWorkflowComponent.class.getName() + "$failConditions";
    
    /** Constant for storing the workflow validation in result map. */
    public static final String WORKFLOW_VALIDATION_KEY = "workflowValidation";
    
    /** Service manager available to subclasses. */
    protected ServiceManager _manager;
    /** Logger available to subclasses. */
    protected Logger _logger;
     
    @Override
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
    }
    
    /**
     * Retrieve the user responsible of the call.
     * @param transientVars the parameters from the call.
     * @return the user responsible of the call.
     * @throws WorkflowException if the user is not present.
     */
    public UserIdentity getUser(Map transientVars) throws WorkflowException
    {
        WorkflowContext workflowContext = (WorkflowContext) transientVars.get("context");

        if (workflowContext == null)
        {
            throw new WorkflowException("Unable to get the workflow context");
        }

        return UserIdentity.stringToUserIdentity(workflowContext.getCaller());
    }

    /**
     * Retrieve the workflow parameters map, or an empty map if there is no parameters
     * @param transientVars the parameters from the call.
     * @return the workflow parameters map
     */
    public Map<String, Object> getContextParameters(Map transientVars)
    {
        return Optional.ofNullable(transientVars.get(CONTEXT_PARAMETERS_KEY))
            .filter(Map.class::isInstance)
            .map(Map.class::cast)
            .orElseGet(() -> new HashMap<>());
    }
    
    /**
     * Retrieve the list condition failure
     * @param transientVars the parameters from the call.
     * @return the list of failed condition
     */
    @SuppressWarnings("unchecked")
    public List<ConditionFailure> getConditionFailures (Map transientVars)
    {
        return (List<ConditionFailure>) transientVars.get(FAIL_CONDITIONS_KEY);
    }
    
    /**
     * Retrieve the map to write return values
     * @param transientVars the parameters from the call.
     * @return the map to render.
     * @throws WorkflowException if the result map is not found.
     */
    @SuppressWarnings("unchecked")
    public Map<String, Object> getResultsMap(Map transientVars) throws WorkflowException
    {
        Map<String, Object> resultMap = (Map<String, Object>) transientVars.get(RESULT_MAP_KEY);
        
        if (resultMap == null)
        {
            throw new WorkflowException("Unable to retrieve result map");
        }
        
        // Found in transient variables map
        return resultMap;
    }
    
    /**
     * Add a global workflow error such as fail condition
     * @param transientVars the parameters from the call.
     * @param errorLabel The error label
     * @throws WorkflowException if the result map is not found.
     */
    public void addWorkflowError (Map transientVars, I18nizableText errorLabel) throws WorkflowException
    {
        Map<String, Object> resultMap = getResultsMap(transientVars);
        
        ValidationResult validationResult = (ValidationResult) resultMap.computeIfAbsent(WORKFLOW_VALIDATION_KEY, __ -> new ValidationResult());
        validationResult.addError(errorLabel);
    }
    
    /**
     * Add a global workflow errors that will be traited as warnings
     * @param transientVars the parameters from the call.
     * @param warnLabel The warning label
     * @throws WorkflowException if the result map is not found.
     */
    public void addWorkflowWarning (Map transientVars, I18nizableText warnLabel) throws WorkflowException
    {
        Map<String, Object> resultMap = getResultsMap(transientVars);
        
        ValidationResult validationResult = (ValidationResult) resultMap.computeIfAbsent(WORKFLOW_VALIDATION_KEY, __ -> new ValidationResult());
        validationResult.addWarning(warnLabel);
    }
    
    /**
     * Record representing a condition failure
     * @param text the failure text
     * @param type the type of failure
     */
    public record ConditionFailure(String text, String type) { /** */ }
}
