/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workflow.readers;

import java.util.List;

import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.workflow.dao.WorkflowStepDAO;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.ConditionalResultDescriptor;
import com.opensymphony.workflow.loader.ResultDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * Serialize PlantUML state diagram
 */
public class PlantUMLSVGReader extends AbstractPlantUMLSVGReader 
{
    @Override
    protected String _getPlantUMLType()
    {
        return "uml";
    }

    @Override
    protected String _getPlantUMLStyle()
    {
        StringBuilder style = new StringBuilder();
        
        style.append("skinparam HyperLinkUnderline false \n");
        style.append("skinparam HyperLinkColor #0a7fb2 \n");
        style.append("skinparam State { \n");
        style.append("BorderThickness 1.5 \n");
        style.append("} \n");
        style.append("hide empty description\n");
        
        return style.toString();
    }

    @Override
    protected String _getPlantUMLGraphContent(Request request, WorkflowDescriptor workflowDescriptor)
    {
        List<StepDescriptor> steps = workflowDescriptor.getSteps();
        StringBuilder content = new StringBuilder();
        if (steps.isEmpty())
        {
            content.append("title \n");
            content.append(_i18nUtils.translate(new I18nizableText("plugin.workflow", "PLUGIN_WORKFLOW_PREVIEW_EMPTY_WORKFLOW")) + " \n");
            content.append("end title \n");
        }
        else
        {
            for (StepDescriptor step : steps)
            {
                //create state node
                content.append(_createState(workflowDescriptor, step));
                
                List<ActionDescriptor> actions = step.getActions();
                for (ActionDescriptor action : actions)
                {
                    //create state's outgoing transitions
                    content.append(_createTransition(workflowDescriptor, step, action));
                }
            }
            
            //create workflow's initials actions
            List<ActionDescriptor> initialActions = workflowDescriptor.getInitialActions();
            for (ActionDescriptor initialAction: initialActions)
            {
                ResultDescriptor unconditionalResult = initialAction.getUnconditionalResult();
                String workflowName = workflowDescriptor.getName();
                String stepLabel = _i18nHelper.translateKey(workflowName, _workflowStepDAO.getStepLabel(workflowDescriptor, 0), WorkflowStepDAO.DEFAULT_STEP_NAME);
                stepLabel = _getStringWithEscapedSpace(stepLabel);
                String actionLabel = _workflowTransitionDAO.getActionLabel(workflowName, initialAction);
                actionLabel = _getStringWithEscapedSpace(actionLabel);
                
                String actionContent = _getTransitionArrow(
                        "[*]",
                        String.valueOf(unconditionalResult.getStep()),
                        _getActionLabel(workflowName, initialAction),
                        _workflowTransitionDAO.getActionIconPathAsBase64(workflowName, initialAction),
                        _getJsFunction(workflowName, "0", stepLabel, String.valueOf(initialAction.getId()), actionLabel),
                        _getActionTooltip(initialAction),
                        false
                        );
                content.append(actionContent);
            }
        }
        return content.toString();
    }
    
    /**
     * Get the transition arrow between 2 states
     * @param incomingState state's id before transition
     * @param outgoingState state's id after transition
     * @param label the transition's label
     * @param iconPath the transition's icon path
     * @param jsFunction the js function for sending current transition as selection
     * @param linkTooltip the transition link's tooltip
     * @param isDashed true if the arrow must be dashed
     * @return the transition arrow content
     */
    protected String _getTransitionArrow(String incomingState, String outgoingState, String label, String iconPath, String jsFunction, String linkTooltip, boolean isDashed)
    {
        StringBuilder nodeContent = new StringBuilder(incomingState);
        nodeContent.append(isDashed ? " -[dashed]-> " : " --> ");
        nodeContent.append(outgoingState);
        if (StringUtils.isNotBlank(label))
        {
            nodeContent.append(": ");
            nodeContent.append(" [[javascript:parent." + jsFunction + " ");
            nodeContent.append("{" + linkTooltip + "}");
            
            if (StringUtils.isNotBlank(iconPath))
            {
                nodeContent.append(" <img:" + iconPath + ">");
            }
            
            nodeContent.append(" " + label);
            nodeContent.append("]]");
        }
        nodeContent.append(" \n");
        
        return nodeContent.toString();
    }
    
    /**
     * Create new state in diagram
     * @param workflow unique id of current workflow
     * @param step a step of the workflow
     * @return a string defining a plantUML state
     */
    protected String _createState(WorkflowDescriptor workflow, StepDescriptor step) 
    {
        String stepToUML = "";
        
        String steplabel = _getStepNodeLabel(workflow, step.getId());
        String stepLabelForJs = _i18nHelper.translateKey(workflow.getName(), _workflowStepDAO.getStepLabel(workflow, step.getId()), WorkflowStepDAO.DEFAULT_STEP_NAME);
        stepLabelForJs = _getStringWithEscapedSpace(stepLabelForJs);
        String elementAbsoluteIconPath = _workflowStepDAO.getStepIconPathAsBase64(workflow, step.getId());
        String stepId = String.valueOf(step.getId());
        String toolip = " {" + _getStepTooltip(step)  + "}";
        String icon = StringUtils.isBlank(elementAbsoluteIconPath)
                ? StringUtils.EMPTY
                : "<img:" + elementAbsoluteIconPath + "> ";
        String jsFunction = " [[javascript:parent." + _getJsFunction(workflow.getName(), stepId, stepLabelForJs, null, null)  + toolip + "]]";
        stepToUML = "state \"" + icon + steplabel + "\" as " + stepId + jsFunction + __MAIN_STEP_NODE_COLOR + " \n";
        
        return stepToUML;
    }
    
    /**
     * Create a new transition between two states in diagram
     * @param workflow unique id of current workflow
     * @param step the initial step of this transition
     * @param transition the current action
     * @return a string defining a plantUML transition
     */
    protected String _createTransition(WorkflowDescriptor workflow, StepDescriptor step, ActionDescriptor transition)
    {
        StringBuilder transitionToUML = new StringBuilder();
        
        String workflowName = workflow.getName();
        String transitionlabel = _getActionLabel(workflowName, transition);
        
        String stepLabel = _i18nHelper.translateKey(workflow.getName(), _workflowStepDAO.getStepLabel(workflow, step.getId()), WorkflowStepDAO.DEFAULT_STEP_NAME);
        stepLabel = _getStringWithEscapedSpace(stepLabel);
        String actionLabel = _workflowTransitionDAO.getActionLabel(workflowName, transition);
        actionLabel = _getStringWithEscapedSpace(actionLabel);
        
        String tooltip = _getActionTooltip(transition);
        String elementAbsoluteIconPath = _workflowTransitionDAO.getActionIconPathAsBase64(workflowName, transition);
        String jsFunction = _getJsFunction(workflow.getName(), String.valueOf(step.getId()), stepLabel, String.valueOf(transition.getId()), actionLabel);
        String initialStep = String.valueOf(step.getId());
        ResultDescriptor unconditionalResult = transition.getUnconditionalResult();
        String finalStep = unconditionalResult.getStep() != -1 
                ? String.valueOf(unconditionalResult.getStep())
                : initialStep;
        
        List<ConditionalResultDescriptor> conditionalResults = transition.getConditionalResults();
        if (!conditionalResults.isEmpty())
        {
            //create intermediary conditional state
            String choiceStateName = "step" + initialStep + "action" + String.valueOf(transition.getId());
            transitionToUML.append("state " + choiceStateName + " <<choice>> \n");
            //create intermediary transition
            String actionContent = _getTransitionArrow(initialStep, choiceStateName, transitionlabel, elementAbsoluteIconPath, jsFunction, tooltip, true);
            transitionToUML.append(actionContent);
            
            //create unconditional transition
            String unconditionalActionContent = _getTransitionArrow(choiceStateName, finalStep, null, null, null, null, false);
            transitionToUML.append(unconditionalActionContent);
            
            //create conditionnal transitions
            for (ConditionalResultDescriptor result : conditionalResults)
            {
                String conditionalStep = result.getStep() != -1 
                        ? String.valueOf(result.getStep())
                        : initialStep;
                if (!conditionalStep.equals(finalStep)) //prevent duplicate transition arrow in graph
                {
                    String conditionalActionContent = _getTransitionArrow(choiceStateName, conditionalStep, null, null, null, null, false);
                    transitionToUML.append(conditionalActionContent);
                }
            }
        }
        else
        {
            //create unconditional transition
            String actionContent = _getTransitionArrow(initialStep, finalStep, transitionlabel, elementAbsoluteIconPath, jsFunction, tooltip, false);
            transitionToUML.append(actionContent);
        }
        return transitionToUML.toString();
    }
}
