/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

 /**
 * Class managing the "add node" action.
 * @private
 */
Ext.define('Ametys.repository.maintenance.MaintenanceActions',
{
	singleton: true,

	/**
	 * Action to start the Garbage Collector task
	 */
	startGarbageCollector: function(controller)
	{
		var taskName = "DATA_STORE_GARBAGE_COLLECTOR";
		this.act(taskName);
	},

	/**
	 * Action to start the Reindexing task
	 */
	reindexing: function(controller)
	{
		var taskName = "REINDEXING";
		this.act(taskName);
	},

	/**
	 * Action to start the Consistency Check task
	 */
	consistencyCheck: function(controller)
	{
		var taskName = "CONSISTENCY_CHECK";
		this.act(taskName);
	},

	/**
	 * @private
	 * Start a maintenance task after asking confirmation
	 * @param {String} taskName The task name
	 */
	act: function(taskName)
	{
		// Check if the task is already running
		Ametys.data.ServerComm.callMethod({
			role: "org.ametys.workspaces.repository.maintenance.MaintenanceTaskComponent",
			methodName: "isRunning",
			parameters: [taskName],
			callback: {
				scope: this,
				handler: this._isRunningCb,
				arguments: {taskName: taskName}
			},
			errorMessage: {
				msg: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_IS_TASK_RUNNING_ERROR_LOG}}",
				category: 'Ametys.repository.maintenance.MaintenanceActions'
			},
			waitMessage: true
		});
	},

	/**
	 * Callback after starting a task
	 * @param {Object} response The response from the server
	 * @param {Object} args The arguments of the callMethod function
	 */
	_isRunningCb: function(response, args)
	{
		var taskName = args.taskName;
		if (response.running == "true")
		{
			this._openTool(taskName);
			return;
		}

		var me = this;
        
        var taskLabel = '';
        switch (taskName) {
            case 'DATA_STORE_GARBAGE_COLLECTOR': taskLabel = "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_TOOL_TITLE_DATA_STORE_GARBAGE_COLLECTOR}}"; break;
            case 'REINDEXING': taskLabel = "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_TOOL_TITLE_REINDEXING}}"; break;
            case 'CONSISTENCY_CHECK': taskLabel = "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_TOOL_TITLE_CONSISTENCY_CHECK}}"; break;
        }
        
		Ametys.Msg.confirm (
            "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_CONFIRM_STARTING_TASK_TITLE}}" + taskLabel, 
		    "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_CONFIRM_STARTING_TASK}}",
		    function (answer)
		    {
				if (answer == 'yes')
				{
					// Start the tasks
					Ametys.data.ServerComm.callMethod({
						role: "org.ametys.workspaces.repository.maintenance.MaintenanceTaskComponent",
						methodName: "startTask",
						parameters: [taskName],
						callback: {
							scope: me,
							handler: me._startTaskCb,
							arguments: {taskName: taskName}
						},
						errorMessage: {
							msg: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_LAUNCH_TASK_BAD_RESPONSE_LOG}}",
							category: 'Ametys.repository.maintenance.MaintenanceActions'
						},
						waitMessage: true
					});
				}
		    }
	    );
	},

	/**
	 * Callback after starting a task
	 * @param {Object} response The response from the server
	 * @param {Boolean} response.launched Is the task launched
	 * @param {Boolean} response.running Is the task running
	 * @param {Object} args The arguments of the callMethod function
	 */
	_startTaskCb: function(response, args)
	{
		if (!response.launched || !response.running)
		{
            Ametys.Msg.show({
                msg: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_LAUNCH_TASK_BAD_RESPONSE_LOG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });

			return;
		}

		this._openTool(args.taskName);
	},

	/**
	 * @private
	 * Open the tool corresponding to the task
	 * @param {String} taskName the name of the task
	 */
	_openTool: function(taskName)
	{
		Ametys.tool.ToolsManager.openTool('uitool-repository-maintenancetask', {'taskName': taskName});
	}
});
