/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Button controller for maintenance tasks.
 * Maintenance tasks button should be disabled by default if the application is not in safe mode or the configuration is incomplete
 */
Ext.define('Ametys.repository.maintenance.MaintenanceTaskButtonController', {
    extend: 'Ametys.ribbon.element.ui.ButtonController',
    
    /**
     * @cfg {Boolean/String} unavailable=false When 'true', indicates that the action is unavailable
     */
    /**
     * @cfg {String} [unavailable-description] The description with the unavailable explanation
     */
    
    /**
     * @property {String} _maintenanceTaskRunning True when a maintenance task is running
     * @private
     */
    
    constructor: function(config)
    {
        var unavailable = config['unavailable'] === true || config['unavailable'] === 'true';
        
        config.disabled = unavailable;
        
        this.callParent(arguments);
        
        if (unavailable)
        {
        	this.setAdditionalDescription(config['unavailable-description']);
        }
        
        // Bus messages listeners
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._disableOnTask, this);
    },
    
    /**
     * Called on each modified message. Looks for
     * {@link Ametys.message.MessageTarget#APPLICATION} message target, and
     * disable the button if status 'MAINTENANCE_TASK_RUNNING' is present.
     * @param {Ametys.message.Message} message The bus message.
     * @private
     */
    _disableOnTask: function(message)
    {
        var target = message.getTarget(Ametys.message.MessageTarget.APPLICATION, 1),
            status;
        if (target)
        {
            status = target.getParameters().status.current || [];
            
            // Task running
            if (this._maintenanceTaskRunning && Ext.Array.contains(status, 'MAINTENANCE_TASK_RUNNING'))
            {
                this.disable();
                this.setAdditionalDescription("{{i18n PLUGINS_REPOSITORYAPP_BUTTON_MAINTENANCE_MAINTENANCE_TASK_RUNNING_DESC}}");
                this._maintenanceTaskRunning = false;
            }
            // Task ended
            else if (!this._maintenanceTaskRunning && !Ext.Array.contains(status, 'MAINTENANCE_TASK_RUNNING'))
            {
                this.enable();
                this.setAdditionalDescription('');
                this._maintenanceTaskRunning = true;
            }
        }
    }
});

