/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
/**
 * Singleton class displaying repository maintenance messages below the ribbon
 */
Ext.define('Ametys.repository.controller.MaintenanceRibbonMessages', {
    singleton: true,
    
    /**
     * @property {String} _repositoryUnavailableId The id of the repository unavailable ribbon message that is currently displayed.
     * @private
     */
    /**
     * @property {String} _maintenanceTaskRunningId The id of the maintenance task running ribbon message that is currently displayed.
     * @private
     */
    /**
     * @property {String} _maintenanceTaskEndedId The id of the maintenance task ended ribbon message that is currently displayed.
     * @private
     */
    
    initMessages: function()
    {
        this._repositoryUnavailableId = null;
        this.maintenanceTaskRunningId = null;
        this.maintenanceTaskEndedId = null;
        
        // Bus messages listeners
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._updateMessages, this);
    },
    
    /**
     * Updates ribbon messages depending on application state
     * @param {Ametys.message.Message} message The bus message.
     * @private
     */
    _updateMessages: function(message)
    {
        var target = message.getTarget(Ametys.message.MessageTarget.APPLICATION, 1),
            status;
            
        if (target)
        {
            status = target.getParameters().status.current || [];
            
            // Repo unavailable
            if (!this._repositoryUnavailableId && Ext.Array.contains(status, 'REPOSITORY_UNAVAILABLE'))
            {
                // Store previous state before disabling
                this._repositoryUnavailableId = Ext.getCmp('ribbon').addMessage({
                    title: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_RIBBON_REPOSITORY_UNAVAILABLE_TITLE}}",
                    type: 'warning',
                    text: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_RIBBON_REPOSITORY_UNAVAILABLE_TEXT}}",
                    closeable: false
                });
            }
            // Repo became available
            else if (this._repositoryUnavailableId && !Ext.Array.contains(status, 'REPOSITORY_UNAVAILABLE'))
            {
                Ext.getCmp('ribbon').closeMessage(this._repositoryUnavailableId);
                this._repositoryUnavailableId = null;
            }
            
            
            // Maintenance task running
            if (!this.maintenanceTaskRunningId && Ext.Array.contains(status, 'MAINTENANCE_TASK_RUNNING'))
            {
                // Store previous state before disabling
                this.maintenanceTaskRunningId = Ext.getCmp('ribbon').addMessage({
                    title: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_RIBBON_MAINTENANCE_TASK_RUNNING_TITLE}}",
                    type: 'info',
                    text: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_RIBBON_MAINTENANCE_TASK_RUNNING_TEXT}}",
                    closeable: true
                });
            }
            else if (this.maintenanceTaskRunningId && !Ext.Array.contains(status, 'MAINTENANCE_TASK_RUNNING'))
            {
                Ext.getCmp('ribbon').closeMessage(this.maintenanceTaskRunningId);
                this.maintenanceTaskRunningId = null;
            }
            
            // Maintenance task ended
            if (!this.maintenanceTaskEndedId && Ext.Array.contains(status, 'MAINTENANCE_TASK_ENDED'))
            {
                // Store previous state before disabling
                this.maintenanceTaskEndedId = Ext.getCmp('ribbon').addMessage({
                    title: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_RIBBON_MAINTENANCE_TASK_ENDED_TITLE}}",
                    type: 'info',
                    text: "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_RIBBON_MAINTENANCE_TASK_ENDED_TEXT}}",
                    closeable: true
                });
                
                // Show 
                Ametys.Msg.confirm (
                    "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_TASK_ENDED_TITLE}}", 
                    "{{i18n PLUGINS_REPOSITORYAPP_MAINTENANCE_TASK_ENDED_TEXT}}",
                    function (answer) {
                        if (answer == 'yes') {
                            Ametys.plugins.admin.restart.RestartActions.restart();
                        }
                    }
                );
            }
            else if (this.maintenanceTaskEndedId && !Ext.Array.contains(status, 'MAINTENANCE_TASK_ENDED'))
            {
                Ext.getCmp('ribbon').closeMessage(this.maintenanceTaskEndedId);
                this.maintenanceTaskEndedId = null;
            }
        }
    }
});
