/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * This class is a singleton class to helper for the script tool
 * @private
 */
Ext.define('Ametys.plugins.coreui.script.ScriptToolHelper', {
    singleton: true,
    
    /**
     * @private
     * @property {Ext.Template} _scriptDateTpl HTML fragment template used for displaying script duration and datetime
     */
    _scriptDateTpl: Ext.create('Ext.Template', '<div class="result-header"><span>',
            "{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_RESULT_MSG}}",
            '{date} ({duration}',
            " {{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_RESULT_MSG_SEC}}",
            ')</span></div>'),
            
       
    /**
     * @private
     * @property {Ext.Template} _helpTooltipTpl HTML fragment template used for help display in tooltip
     */
    _helpTooltipTpl: Ext.create('Ext.XTemplate', 
            "<p>{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_SCRIPT_HINT}}</p>",
            "<tpl if='variables'>",
                "<p>{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_SCRIPT_HINT_VARIABLES}}<ul>",
                "<tpl foreach='variables'>",
                    "<li><b>{$}</b> : {.}</li>",
                "</tpl></ul></p>",
            "</tpl>",
            "<tpl if='functions'>",
                "<p>{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_SCRIPT_HINT_FUNCTIONS}}<ul>",
                "<tpl foreach='functions'>",
                    "<li><b>{$}</b> : {.}</li>",
                "</tpl></ul></p>",
            "</tpl>",
            "<p>{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_SCRIPT_HINT_ADDITIONAL_SCRIPT}}</p>"), 
    

    /**
     * Format the console output
     * @param {Object} result The script result to format
     * @return {String} the result to HTML format
     */
    formatConsoleOutput: function (result) 
    {
        var id = Ext.id();
        var html = "<div id='" + id + "'>";

        var start = Ext.isDate(result.start) ? result.start : new Date(result.start);
        var end = Ext.isDate(result.end) ? result.end : new Date(result.end);
        
        var duration = (end.getTime() - start.getTime()) / 1000.0;
        
        if (!Ext.isEmpty(result.output))
        {
            html += '<pre>' + result.output + '</pre>';
        }
        if (!Ext.isEmpty(result.result))
        {
            html += Ext.JSON.prettyEncode(result.result, 0, function(value, offset) {
                if (Ext.isObject(value) && value.type == 'node' && value.path != null)
                {
                    return value.path;
                }
                if (Ext.isObject(value) && value.type == 'ametys-object' && value.path != null)
                {
                    return value.path + " [" + value.id + "]";
                }
            });
        }
        if (!Ext.isEmpty(result.error))
        {
            html += '<div class="error">' + result.error + '</div>';
        }
        if (Ext.isEmpty(result.output) && Ext.isEmpty(result.result) && Ext.isEmpty(result.error))
        {
            html += "<div class='no-result'>{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_RESULTS_NO_RESULT}}</div>";
        }
        if (!Ext.isEmpty(result.stacktrace))
        {
            html += '<div class="fatalerror">' + result.message
                    + "<br/><a href='javascript:void()' onclick='this.nextSibling.style.display = (this.nextSibling.style.display == \"none\" ? \"\" : \"none\")'>{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_ERROR_DETAILS}}</a>"
                    + "<div class='stacktrace' style='display: none'>" + result.stacktrace + "</div>"
                    + '</div>';
        }
        
        html += this._scriptDateTpl.apply ({date: Ext.Date.format(end, Ext.Date.patterns.ShortTime), duration: duration});
        html += '</div>';

        return html;
    },
    
    /**
     * Initialize the tooltip help of the script tool
     * @param {String|HTMLElement} target The target element or the id of the target element for the tooltip. 
     */
    initializeTooltipHelp: function(target)
    {
        Ametys.data.ServerComm.callMethod({
            role: 'org.ametys.plugins.core.ui.script.ScriptHandler',
            methodName: 'getScriptBindingDescription',
            callback: {
                scope: this,
                handler: this._getScriptBindingDescriptionCb,
                arguments: [target]
            },
            errorMessage: {
                msg: "{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_SCRIPT_HINT_ERROR}}"
            }
        });
    },
    
    /**
     * @private
     * Callback function invoked after retrieving the description of available script functions and variables
     * @param {Object} result The result
     * @param {Object[]} args The callback arguments
     */
    _getScriptBindingDescriptionCb: function(result, args)
    {
        var help = this._helpTooltipTpl.applyTemplate(result);
        
        Ext.create('Ext.tip.ToolTip', {
            target: args[0],
            title: "{{i18n PLUGINS_CORE_UI_TOOLS_SCRIPT_HELP_TITLE}}",
            html: help,
            cls: 'x-fluent-tooltip',
            showDelay: 0,
            dismissDelay: 0
        });
    }
    

});
