/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class is a configurable form panel that can contains tabs, fieldsets, repeaters and widgets. Configuration is made through XML or JSON requests.
 * The configuration format can be in JSON or XML.
 * The 2 steps to use this components are to call once and only once: 
 *  
 * 1) create the form (#configure) 
 * 2) fill the values (#setValues)
 */
Ext.define('Ametys.form.ConfigurableFormPanel2018', {
    extend: "Ametys.form.ConfigurableFormPanel",
    
    /**
     * @cfg {String} defaultPathSeparator='/' The default separator for fields
     */
    defaultPathSeparator: '/',
    
    /**
     * @method configure
     * 
     * This function creates and add form elements from a definition
     * 
     * The JSON configuration format is
     * 
     *     {
     *         "<fieldName>": {
     *             "label":        "My field"
     *             "description":  "This describes what my field is made for"
     *             "type":         "STRING",
     *             "validation": {
     *                 "mandatory":    true
     *             },
     *             "multiple":       false
     *         },
     *         // ...
     *         "groups": [{
     *             "role": "tab"
     *             "label": "My first tab",
     *             "description": "Lorem ipsum dolor sit amet",
     *             "descriptionCls": ['my-custom-css-class'],
     *             "elements": {
     *                 "<fieldName>": {...},
     *                 "groups": [{
     *                     "role": "fieldset",
     *                     "label": "My fieldset",
     *                     "description": "This describes what my composite is made for",
     *                     "elements": {...}
     *                 },
     *                 {
     *                     "type": "composite",
     *                     "label": "My composite",
     *                     "description": "This describes what my composite is made for",
     *                     "elements": {...}
     *                 },
     *                 {
     *                     "type": "repeater",
     *                     "label": "My repeater",
     *                     "description": "This describes what my repeater is made for",
     *                     "add-label": "Add",
     *                     "del-label": "Delete",
     *                     "header-label": "Some header",
     *                     "min-size": 0,
     *                     "min-size": 5,
     *                     "initial-size": 1,
     *                     "elements": {...}
     *                 }]
     *                 // ...
     *             }
     *         },
     *         {
     *             "role": "tab",
     *             // ...
     *         }],
     *         "field-checker": [{
     *             "id": "my-checker-id",
     *             "label": "My field checker",
     *             "description": "This describes what my field checker is made for",
     *             "icon-glyph": "ametysicon-data110",
     *             "linked-field": ['linked.field.1.path', 'linked.field.2.path'], // you can begin a linked field with the default separator (#cfg-defaultPathSeparator),
     *                                                                             // in order to use an absolute path instead of a relative path, see the
     *                                                                             // second field checker below where we have defaultPathSeparator = "/"
     *             "order": "1" // used to sort the field checkers amongst themselves, the smaller the vertically higher
     *         }, 
     *         { 
     *             "id": "my-other-checker-id",
     *             "label": "My second field checker",
     *             "description": "This describes what my second field checker is made for",
     *             "small-icon-path": "path for the small icon representing the test",
     *             "medium-icon-path": "path for the medium icon representing the test",
     *             "large-icon-path": "path for the large icon representing the test",
     *             "linked-field": ['/linked.field.1.path', '/linked.field.2.path'],
     *             "order": "10", // will be displayed below the first checker 
     *         }]
     *     }
     * 
     * The **&lt;fieldName&gt;** is the form name of the field. (Note that you can prefix all field names using #cfg-fieldNamePrefix). See under for the reserved fieldName "groups"
     * 
     * The String **label** is the readable name of your field that is visible to the user.
     * 
     * The String **description** is a sentence to help the user understand the field. It will appear in a tooltip on the right help mark.
     * 
     * The String **type** is the kind of value handled by the field. The supported types depend on the configuration of your Ametys.runtime.form.WidgetManager. Kernel provides widgets for the following types (case is not important):
     * STRING, LONG, DOUBLE, BOOLEAN, DATE, DATETIME, BINARY, FILE, GEOCODE, REFERENCE, RICH_TEXT, USER.
     * 
     * The boolean **labelWithField** can be set to true for BOOLEAN types using edition.checkbox widget in order to display the label on the right of the checkbox instead of on the left. It can also be put in the object **widget-params**.
     * 
     * The object **validation** field is a field validator.
     * Can be an object with the optional properties 
     * 
     * - boolean **mandatory** to true, to check the field is not empty AND add a '*' at its label.
     * - String **invalidText** : a general text error if the field is not valid
     * - String **regexp** : a regular expression that will be checked
     * - String **regexText** : the text error if the regexp is not checked
     * 
     * The Object array **enumeration** lists available values. Note that types and widgets that can be used with enumeration is quite limited.
     * Each item of the array is an object with **value** and **label**.
     * Example: enumeration: [{value: 1, label: "One"}, {value: 2, label: "Two"}]
     * 
     * The Object **default-value** is the default value for the field if not set with #setValues.
     * 
     * The boolean **multiple** specifies if the user can enter several values in the field. Types and widgets that support multiple data is quite limited.
     * 
     * The String **widget** specifies the widget to use. This is optional to use the default widget for the given type, multiple and enumeration values.
     * The widgets are selected using the js class {@link Ametys.form.WidgetManager} and the extension point org.ametys.runtime.ui.widgets.WidgetsManager.
     * Note that you can transmit additional configuration to all widgets using #cfg-additionalWidgetsConfFromParams 
     * 
     * The optional Object **widget-params** will be transmitted to the widget configuration : values depends on the widget you did select.
     * 
     * The boolean **hidden** will hide this field.
     * 
     * The boolean **can-not-write** makes the field in read only mode.
     * 
     * The Object array **annotations** describes available XML annotations on a richtext.
     * Each item is an object with properties : **name** (the XML tagname), **label** (the label of the button to set this annotation, defaults to name) and **description** (the help text associated to the button).
     * Exemple: annotations: [ { name: "JUSTICE", label: "Justice term", description: "Use this button to annotate the selected text as a justice term" } ] 
     * 
     * The object **disableCondition** can be defined in order to disable/enable the current parameter. It has the following configuration that must be written in JSON:
     *
     * - Object **conditions** conditions that can contain several condition objects or other conditions
     *   - Object **conditions** recursively describe sub conditions groups.
     *   - Object[] **condition** Object describing a unit condition (see under).
     *   - String **type** the type of the underlying conditions. Can be set to "and" (default value) or "or".
     * 
     * The Object **condition** has the following attributes:
     * 
     * - String **id** the id of the field that will be evaluated
     * - String **operator** the operator used to evaluated the field. Can be **eq**, **neq**, **gt**, **geq**, **leq** or **lt**         
     * - String **value** the value with which the field will be compared to
     *
     * The object **field-checker** can be used in order to check the value of certain fields. It can contain the following attributes: 
     *
     *  - String **id** The id of the parameter checker. 
     *  - String **large-icon-path** The relative path to the 48x48 icon representing the test
     *  - String **medium-icon-path** The relative path to the 32x32 icon representing the test
     *  - String **small-icon-path** The relative path to the 16x16 icon representing the test
     *  - String **icon-glyph** the glyph used for the icon representing the test
     *  - String **icon-decorator** the decorator to use on the icon
     *  - String[] **linked-fields** the absolute or relative paths of the linked field (fields used to run the check). Always JSON-encoded even for XML configurations.
     *  - String **label** The label of the parameter checker
     *  - String **description** The description of the parameter checker
     *  - String **order** The displaying order (from top to bottom). Indeed, several parameter checkers can be defined on a given tab, fieldset or parameter.
     *  
     * You can define a **field-checker** directly under the data node/object, in which case the checker will be global and displayed at the bottom of 
     * the main form container. See the example at the top of this documentation.
     * 
     * 
     * The **fieldname "groups"** is a reserved keyword to create a graphical or logical grouping of fields. It can contain the following attributes:
     * 
     *  - String **label** The label of the grouping.
     *  - String **description** The description of the grouping (as HTML). It is inserted at the beginning of the elements of the fieldset.
     *  - String[] **descriptionCls** An optional array of extra CSS classes to add to the text component's element of the description.
     *  - Object **field-checker** (see above) Groups can also have a parameter checker for both "tab" and "fieldset" role
     *  - Object **elements** The child elements of the group: this is a recursive data object, except that **"groups"** can not be used again with the role "tab".
     *  
     * A graphical grouping of fields must have the **role** attribute. It can be "tab" or "fieldset" to create a tab grouping or a fieldset grouping. Note that tab grouping can be replaced by simple panels according to a user preference.
     *  
     * A logical grouping of fields, unlike the graphical ones, will have impact on the children field paths. It must have a **type** attribute that can be "composite" or "repeater".
     * A composite can have a **switcher** attribute. The switcher is a boolean field that is used to enable/disable the other fields of its group. The switcher object must have a **label** and an **id** (the name of the boolean field) and optionally a **default-value**.
     * A repeater needs the following attributes:
     * 
     * - String **add-label**: The label to display on the add button
     * - String **del-label**: The label to display on the delete button
     * - String **header-label**: The label to display on the repeater itselft
     * - Number **min-size**: The optional minimum size of the repeater. For example 2 means it will at least be repeated twice. 0 if not specified.
     * - Number **max-size**: The optional maximum size of the repeater. Default value is infinite.
     * - Number **initial-size**: The optional size when loading the form (must be between minSize and maxSize). minSize is the default value.
     * 
     * @param {Object} data The data to create the form structure.
     */
    
    /**
     * @private
     * This function creates and add form elements from a JSON definition
     * @param {Object} data The JSON definition of the form fields.
     * @param {String} prefix The input prefix to concatenate to input name
     * @param {Ext.container.Container} [ct=this] The container where to add the form elements
     * @param {Number} [offset=0] The field offset.
     * @param {Number} [roffset=0] The field right offset.
     */
    _configureJSON: function (data, prefix, ct, offset, roffset)
    {
        prefix = prefix || this.getFieldNamePrefix();
        ct = ct || this.getFormContainer();
        offset = offset || 0;
        roffset = roffset || 0;
        
        var tabs = [];
        this._hasFieldsBeforeTabs = false;
        
        for (var name in data)
        {
            if (!data[name])
            {
                continue;
            }
            
            var nestingLevel = this._getNestingLevel(ct);
            
            if (name == 'groups')
            {
                for (var i=0; i < data[name].length; i++)
                {
                    var group = data[name][i];
                    var type = group.type ? group.type.toLowerCase() : null;
                    
                    if (group.role == 'tab' && nestingLevel == 1)
                    {
                        tabs.push(group);
                    }
                    else if (type == 'composite')
                    {
                        this._configureCompositeJSON(group, prefix, ct, offset, roffset, nestingLevel);
                    }
                    else if (type == 'repeater')
                    {
                        this._configureRepeaterJSON(group, prefix, ct, offset, roffset, nestingLevel);
                    }
                    else // fieldset
                    {
                        this._configureFieldsetJSON(group, prefix, ct, offset, roffset, nestingLevel);
                    }
                }
            }
            else if (!data[name].type && name == 'field-checker' && nestingLevel == 1)
            {
                // Global field checker
                 var fieldCheckers = data[name];
                 if (!Ext.isEmpty(fieldCheckers))
                 {
                     this._fieldCheckersManager.addFieldCheckers(ct, fieldCheckers, prefix, offset, roffset);
                 }
            }
            else
            {
                this._configureFieldJSON(data[name], name, prefix, ct, offset, roffset);
            }
        }
        
        this._configureTabsJSON(tabs, prefix, offset, roffset);
    },
    
    /**
     * @private
     * This function creates and add a fieldset from a JSON definition
     * @param {Object} fieldsetCfg The JSON definition of the fieldset.
     * @param {String} prefix The input prefix to concatenate to input name
     * @param {Ext.container.Container} ct The container where to add the fieldset
     * @param {Number} offset The fieldset offset.
     * @param {Number} roffset The fieldset right offset.
     * @param {Number} nestingLevel The fieldset nesting level
     * @param {Object} switcherCfg The configuration object of the group switcher
     */
    _configureFieldsetJSON: function(fieldsetCfg, prefix, ct, offset, roffset, nestingLevel, switcherCfg)
    {
        var fieldset = this._addFieldSet(ct, fieldsetCfg.label, nestingLevel, switcherCfg);
        
        // Transmit offset + 5 (padding) + 1 (border) + 11 (margin + border) if we are in a nested composite.
        var finalOffset = offset 
                        + Ametys.form.ConfigurableFormPanel.HORIZONTAL_PADDING_FIELDSET
                        + (nestingLevel > 1 ? Ametys.form.ConfigurableFormPanel.OFFSET_FIELDSET : 0)
                        + 1;
        var finalROffset = roffset 
                        + Ametys.form.ConfigurableFormPanel.HORIZONTAL_PADDING_FIELDSET
                        + 1;
                        
        this._addDescription(fieldset, fieldsetCfg, finalOffset, finalROffset);
        this._configureJSON(fieldsetCfg.elements, prefix, fieldset, finalOffset, finalROffset);
        
        var fieldCheckers = fieldsetCfg['field-checker'];
        if (!Ext.isEmpty(fieldCheckers))
        {
            this._fieldCheckersManager.addFieldCheckers(fieldset, fieldCheckers, prefix, finalOffset, finalROffset);
        }
    },
    
    /**
     * @private
     * This function creates and add a composite from a JSON definition
     * @param {Object} composite The JSON definition of the composite.
     * @param {String} prefix The input prefix to concatenate to input name
     * @param {Ext.container.Container} ct The container where to add the composite
     * @param {Number} offset The composite offset.
     * @param {Number} roffset The composite right offset.
     * @param {Number} nestingLevel the composite nesting level
     */
    _configureCompositeJSON: function(composite, prefix, ct, offset, roffset, nestingLevel)
    {
        var switcherCfg = composite.switcher;
        if (switcherCfg != null)
        {
            switcherCfg.name = switcherCfg.id;
            delete switcherCfg.id;
            switcherCfg.type = 'boolean';
            switcherCfg.value = switcherCfg.value || switcherCfg['default-value'] || false;
            delete switcherCfg['default-value'];
        }
        
        if (composite.name)
        {
            prefix = prefix + composite.name + this.defaultPathSeparator;
        }
        this._configureFieldsetJSON(composite, prefix, ct, offset, roffset, nestingLevel, switcherCfg);
    },
    
    /**
     * @private
     * This function creates and add a repeater from a JSON definition
     * @param {Object} repeater The JSON definition of the repeater.
     * @param {String} prefix The input prefix to concatenate to input name
     * @param {Ext.container.Container} ct The container where to add the repeater
     * @param {Number} offset The repeater offset.
     * @param {Number} roffset The repeater right offset.
     * @param {Number} nestingLevel the repeater nesting level
     */
    _configureRepeaterJSON: function(repeater, prefix, ct, offset, roffset, nestingLevel)
    {
        var repeaterCfg = {
            prefix: prefix,
            name: repeater.name,
            
            label: repeater.label,
            description: repeater.description,
            
            addLabel: repeater['add-label'],
            delLabel: repeater['del-label'],
            headerLabel: repeater['header-label'],
            
            minSize: repeater['min-size'] || 0,
            maxSize: repeater['max-size'] || Number.MAX_VALUE,
            
            readOnly: repeater['can-not-write'] === true,
            
            composition: repeater.elements,
            
            fieldCheckers: repeater['field-checker'],
            
            nestingLevel: nestingLevel,
            offset: offset,
            roffset: roffset
        }
            
        this._addRepeater (ct, repeaterCfg, repeater['initial-size'] || 0);
    },
    
    /**
     * Creates and returns a repeater panel from the configuration object
     * @param {Object} config The repeater configuration object. See {@link Ametys.form.ConfigurableFormPanel.Repeater2018} configuration.
     * @return {Ametys.form.ConfigurableFormPanel.Repeater2018} The created repeater panel
     * @private
     */
    _createRepeater: function (config)
    {
        var repeaterCfg = Ext.applyIf(config, {
            minSize: 0,
            maxSize: Number.MAX_VALUE,
            defaultPathSeparator: this.defaultPathSeparator,
            form: this
        });
        
        return Ext.create('Ametys.form.ConfigurableFormPanel.Repeater2018', repeaterCfg);
    },
    
    /**
     * @private
     * Retrieves the repeater prefix to use to find the field panel
     * @param {Object} repeater the repeater
     * @return {String} the repeater's prefix
     */
    _getRepeaterPanelPrefix: function(repeater)
    {
        var prefix = this.getFieldNamePrefix() + repeater.prefix;
        return prefix.replace(/\./g, '\\.').replace(/\[|\]/g, '.');
    },
    
    /**
     * @method setValues
     * 
     * Fill the configured form with values. #configure must have been called previously with data matching the configured data.
     * 
     * The data can be a JSON or an XML object.
     * 
     * The JSON format
     * ===============
     * See the following structure:
     *
     *     {
     *         "values": {
     *             "<fieldPath>": "a string value",
     *             // ...
     *         },
     *     
     *         "comments": {
     *             "<fieldPath>": [{
     *                 "text": "My comment\non two lines",
     *                 "author": "Author Fullname",
     *                 "date": "2020-12-31T23:59:59.999+02:00"
     *             }],
     *             // ...
     *         },
     *            
     *         "invalid": {
     *             "otherfield": "rawvalue"
     *             // ...
     *         },
     *              
     *         "repeaters": [{
     *             "prefix": "",
     *             "name": "repeater1",
     *             "count": 2
     *         },
     *         {
     *             "prefix": "repeater1[1]/",
     *             "name": "repeater2",
     *             "count": 2
     *         },
     *         {
     *             "prefix": "repeater1[1]/repeater2[1]/",
     *             "name": "repeater3",
     *             "count": 2
     *         },
     *         {
     *             "prefix": "repeater1[1]/repeater2[2]/",
     *             "name": "repeater3",
     *             "count": 1
     *         },
     *         {
     *             "prefix": "repeater1[2]/",
     *             "name": "repeater2",
     *             "count": 1
     *         },
     *         {
     *             "prefix": "repeater1[2]/repeater2[1]/",
     *             "name": "repeater3",
     *             "count": 2
     *         }]
     *     }
     * 
     * The **values**, **comments** and **invalid** keywords are configurable via the **valuesTagName**, **commentsTagName** and **invalidFieldsTagName** parameter (see below).
     * 
     * The Object **values** will fill the fields. The **&lt;fieldPath&gt;** is the path of the field to fill.
     * Example: {"rootFieldName": "a value", "repeater1[1]/repeater2[2]/otherFieldName": "another value", ...}
     * 
     * The Object **comments** will add comments on fields. The **&lt;fieldPath&gt;** is the path of the field to comment. It is an Object array, one entry per comment.
     * A comment have the following mandatory attributes :
     *   - String **text** The comment itself
     *   - String **date** The date of the comment using the ISO 8601 format (will use the Ext.Date.patterns.ISO8601DateTime parser).
     *   - String **author** The fullname of the author of the comment.
     * 
     * The Object **invalid** will pre-fill fields with raw values. For exemple, you can pre-fill a date field with a non-date string.
     * The **invalid** values should not set the same values already brought by **values**, but they will replace them in such a case.
     *      * 
     * The Object array **repeaters** allows to know the size of every repeater. Each entry has this attributes:
     * 
     * - String **name** The name of the repeater
     * - String **prefix** The path to this repeater
     * - Number **count** The size of the repeater
     * 
     * 
     * The XML format
     * ==============
     * See the following structure:
     * 
     *      <myrootnode>
     *          <values>
     *              <fieldname json="false" value="3"/>
     *              <!-- ... -->
     *          </values>
     *          
     *          <comments>
     *              <field path="fieldname">
     *                  <comment id="1" date="2020-12-31T23:59:59.999+02:00">
     *                      My comment for the field <fieldname>
     *                  </comment>
     *                  <!-- ... -->
     *              </field>
     *              <!-- ... -->
     *          </comments>
     *      </myrootnode>
     * 
     * About the **values**:
     * 
     * - the tag name is the name of the concerned field (without prefix).
     * - those tags are recursive for sub-field (child of composites).
     * - for repeaters, an attribute **entryCount** is set on the tag, its value is the size of the repeater. Each entry is encapsulated in an **entry** tag with an attribute **name** equals to the position (1 based) of the entry.
     * - the attribute **json** set to true means the value will be interpreted as JSON before being set on the field
     * - the value itself can be either the value of the attribute **value**, or the text of the tag
     * - multiple values are set by repeating the tag.
     *   
     * About the **comments**:
     * 
     * - the **fields** are not recursive 
     * - the **path** attribute contains the fieldname (without prefix) with '/' separator for sub-fields (child of composites). For repeaters, you also have to add the position of the repeater to modify. Exemple: path="mycompositefield/repeater[2]/myfield"
     * - the **comment** tag have the following mandatory attributes :
     *   - **id** The number of the comment
     *   - **date** The date of the comment using the ISO 8601 format (will use the Ext.Date.patterns.ISO8601DateTime parser).
     *   - **author** The fullname of the author of the comment.
     *   
     * There is no **invalid** tag available.
     *  
     *  Here is a full example:
     *  
     *      <myrootnode>
     *          <values>
     *              <!-- A simple text value -->
     *              <title>My title</title>
     *              <!-- A composite -->
     *              <illustration>
     *                  <alt-text>My alternative text</alt-text>
     *              </illustration>
     *              <!-- A richtext value -->
     *              <content>&lt;p&gt;my rich text value&lt;/p&gt;</content>
     *              <!-- A repeater -->
     *              <attachments entryCount="1">
     *                  <entry name="1">
     *                      <!-- A file metadata. The widget waits for an object value according to its documentation {@link Ametys.form.widget.File#setValue} -->
     *                      <attachment json="true">
     *                          {
     *                              "type": "metadata",
     *                              "mimeType": "application/unknown",
     *                              "path": "attachments/1/attachment",
     *                              "filename": "ametysv4.ep",
     *                              "size": "188249",
     *                              "lastModified": "2015-06-03T14:15:22.232+02:00",
     *                              "viewUrl": "/cms/plugins/cms/binaryMetadata/attachments/1/attachment?objectId=content://ec7ef7a1-139a-4863-a866-76196ed556cb",
     *                              "downloadUrl": "/cms/plugins/cms/binaryMetadata/attachments/1/attachment?objectId=content://ec7ef7a1-139a-4863-a866-76196ed556cb&amp;&download=true"
     *                          }
     *                      </attachment>
     *                      <attachment-text>fichier</attachment-text>
     *                  </entry>
     *              </attachments>
     *          </values>
     *          
     *          <comments/>
     *      </myrootnode>
     * 
     * @param {Object/HTMLElement} data The object that will fill the form.
     * @param {String} [valuesTagName=values] the tag name for the values 
     * @param {String} [commentsTagName=comments] the tag name for the comments 
     * @param {String} [invalidFieldsTagName=invalid] the tag name for the invalid fields
     * @param {String} [warnIfNullMessage] the warning message to display if there is no value
     */
    
    /* --------------------------------------------------------------------- */
    /*               Helper methods to work on relative fields               */
    /* --------------------------------------------------------------------- */
    
    /**
     * Helper method to get all children fields of a repeater
     * @param {Ametys.form.ConfigurableFormPanel.Repeater2018} repeater The repeater
     * @return {Ext.util.MixedCollection} The children fields of this repeater
     */
    getChildrenFields: function(repeater)
    {
        return this.getForm().getFields().filter('name', this.getFieldNamePrefix() + repeater.name + '[' + repeater._lastInsertItemPosition + ']' + this.defaultPathSeparator);
    }
});
