/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

 /**
  * This factory creates Ametys.message.MessageTarget for a server log category
  * 
  * See #createTargets to know more.
  * @private
  *  
  */
Ext.define("Ametys.plugins.coreui.log.LogCategoryMessageTargetFactory", {
    extend: "Ametys.message.factory.DefaultMessageTargetFactory",
    
        /**
         * Create the targets for a message.
         * You have to choose between the parameters: ids or id+level
         * @param {Object} parameters The parameters needed by the factory to create the message. Can not be null. Handled elements are
         * @param {String[]} parameters.ids The log categories identifier. When specified, must be alone. 
         * @param {String} parameters.id The log category identifier. Must go with level 
         * @param {String} parameters.level The log category level. Must go with id. Can be 'DEBUG', 'INFO'... 
         * @param {Function} callback The callback function called when the targets are created. Parameters are
         * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
         */
        createTargets: function(parameters, callback)
        {
            if (parameters.ids)
            {
                this.serverCall('getLevels', [parameters.ids], this._createTargets, { arguments: callback });
            }
            else 
            {
                var target = Ext.create("Ametys.message.MessageTarget", {
                    id: Ametys.message.MessageTarget.LOG_CATEGORY,
                    parameters: {
                        id: parameters.id,
                        level: parameters.level
                    }
                });
                callback([target]);
            }
        },

        /**
         * Create the targets
         * @param {Object} allLevels An object with levels et effectiveLevels. In each one the category and their associated level
         * @param {Function} callback The callback function called when the targets are created. Parameters are
         * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
         * @private
         */
        _createTargets: function (allLevels, callback)
        {
            var targets = [];
            
            var levelsByCategory = allLevels.levels;
            
            Ext.Object.each(levelsByCategory, function(id, level) {
                target = Ext.create('Ametys.message.MessageTarget', {
                    id: Ametys.message.MessageTarget.LOG_CATEGORY,
                    parameters: {id: id, level: level, effectiveLevel: allLevels.effectiveLevels[id]}
                });
                
                targets.push(target);
            });
            
            callback(targets);
        }    
});
 
Ext.define("Ametys.message.LogCategoryMessageTarget", {
    override: "Ametys.message.MessageTarget",

     statics: 
     {
         /**
          * @member Ametys.message.MessageTarget
          * @readonly
          * @property {String} LOG_CATEGORY The target type is a log category. The expected parameters are:
          * @property {String} LOG_CATEGORY.id The id of the log category
          * @property {String} LOG_CATEGORY.level The level of the log category
          * @property {String} [LOG_CATEGORY.effectiveLevel] The level resolved of the log category. You will not see "inherit" here
          */
         LOG_CATEGORY: "log-category"
     }
});
